/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import java.util.List;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.i18n.Messages;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;

/**
 * Throw when there is some issue with the command that deserves an error
 * message to the user along with potentially a non-zero exit
 */
public class ExitException extends RiskscapeException {

  /**
   * Magic problem that indicates that there wasn't a problem related to this exit exception, such as when the user just
   * wants to quit
   */
  public static final Problem NO_PROBLEM = new Problem(Severity.INFO, "");

  private static final long serialVersionUID = 921444503735663014L;

  private static final int DEFAULT_ERROR_EXIT_CODE = 1;

  private final int exitCode;

  /**
   * Throw an {@link ExitException} that will cause riskscape to exit immediately without error
   */
  public static void quit() {
    throw new ExitException(0, NO_PROBLEM);
  }

  public ExitException(int exitCode, String message, Object... formatArgs) {
    this(exitCode, null, message, formatArgs);
  }

  public ExitException(int exitCode, String message, List<Problem> problems) {
    super(Problem.composite(problems, "%s", message));
    this.exitCode = exitCode;
  }

  public ExitException(int exitCode, Problems problem) {
    super(problem);
    this.exitCode = exitCode;
  }

  public ExitException(Problems problem) {
    this(problem, null);
  }

  public ExitException(Problems problem, Throwable cause) {
    super(problem, cause);
    this.exitCode = DEFAULT_ERROR_EXIT_CODE;
  }

  /**
   * @param exitCode the exit code to exit(int) with
   * @param cause a reason for the exit, can be null
   * @param message a message to print out on to the command line, first argument to
   * {@link String#format(String, Object...)}
   * @param formatArgs parameters to interpolate in the the message, second argument to
   * {@link String#format(String, Object...)}
   */
  public ExitException(int exitCode, Throwable cause, String message, Object... formatArgs) {
    super(String.format(message, formatArgs), cause);
    this.exitCode = exitCode;
  }


  public ExitCode toExitCode(Messages messages) {
    if (getProblem() == NO_PROBLEM) {
      return new ExitCode(exitCode);
    } else {
      return new ExitCode(exitCode, messages.renderProblem(this).toString());
    }
  }
}
