/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Represents an expected property or 'trait' that the parameter should have,
 * especially in terms of how a UI user should specify the parameter value. The
 * user can define these properties in a ParameterTemplate, which then gets
 * applied to an actual {@link Parameter}.
 */
public interface ParameterProperty {

  /**
   * @return the subset of properties that match the given class
   */
  static <T extends ParameterProperty> List<T> filter(Collection<ParameterProperty> properties, Class<T> clazz) {
    return properties.stream().filter(prop -> clazz.isInstance(prop))
        .map(prop -> clazz.cast(prop))
        .collect(Collectors.toList());
  }

  /**
   * @return the keyword used in the project.ini to define this property
   */
  String getKeyword();

  /**
   * The presence of one property can imply other properties, e.g. the 'min'
   * property implies that the parameter must be 'numeric'.
   *
   * @return any other properties that this particular property implies
   */
  List<ParameterProperty> getImplied();

  /**
   * @return true if this property implied that the other given property is also true
   */
  default boolean implies(ParameterProperty other) {
    return getImplied().contains(other);
  }

  /**
   * @return true if this property is defined using a key-value pair, e.g. `min: 0`,
   * false if just a single keyword, e.g. `numeric`.
   */
  default boolean hasKeyValuePair() {
    return getValueOr().isPresent();
  }

  /**
   * Clones this ParameterProperty with the value part of a key-value pair. Only
   * supported if {@link #hasKeyValuePair()} is true.
   */
  default ResultOrProblems<ParameterProperty> withValue(BindingContext context, Object value) {
    throw new UnsupportedOperationException(getClass() + " does not support key-value pairs");
  }

  default ParameterProperty withValue(Object newValue) {
    throw new UnsupportedOperationException(getClass() + " does not support key-value pairs");
  }

  /**
   * @return the type of value associated with this property, if any. E.g. for a key value
   *         pair property, like `min: 0`, it'd return zero (wrapped in an Optional)
   */
  default Optional<?> getValueOr() {
    return Optional.empty();
  }

  /**
   * @return true if another property is compatible with this one or not, e.g.
   *         'min' and 'numeric' are compatible, but 'min' and 'bookmark' are not.
   */
  boolean isCompatible(ParameterProperty other);

  /**
   * Optionally validates that the bound value for the parameter conforms to this
   * property, e.g. for a `min: 0` property it would return an error if -1 were
   * entered
   */
  default List<Problem> validate(BindingContext context, Object bound) {
    return Collections.emptyList();
  }

}

