/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import java.util.List;

import lombok.Getter;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * Thrown by {@link ParameterBinder}s when a value can not be converted to the desired type for some reason.
 * The message will be shown to the user, so make it informative.
 */
@SuppressWarnings("serial")
public class ParameterBindingException extends RiskscapeException {

  @Getter
  private Parameter modelParameter;

  public ParameterBindingException(Parameter parameter, String format, Object... args) {
    super(String.format(format, args));
    this.modelParameter = parameter;
  }

  /**
   * Build a binding exception that wraps a list of Problems, wrapping them in a `foundWith` error that includes the
   * parameter as context
   */
  public ParameterBindingException(Parameter parameter, List<Problem> problems) {
    super(Problems.foundWith(parameter, problems));
    this.modelParameter = parameter;
  }

  public ParameterBindingException(Parameter parameter, Problem problem) {
    super(problem);
    this.modelParameter = parameter;
  }

  /**
   * Extends the default behaviour of getProblem to return either the problem the error was created with, or will
   * concoct one from the message the exception was created with.  This is a half-way house - ultimately all the binding
   * code should be changed to use problems and i18n instead of anonymous messages in exceptions, but this at least
   * allows code to start using `ParameterBindingException.getProblem` instead of `getMessage`
   */
  @Override
  public Problem getProblem() {
    Problem fromSuper = super.getProblem();

    if (fromSuper == null) {
      return Problem.error("%s", getMessage());
    } else {
      return fromSuper;
    }
  }
}
