/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import nz.org.riskscape.config.ConfigString;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;

public interface ParamProblems extends ProblemFactory {

  static ParamProblems get() {
    return Problems.get(ParamProblems.class);
  }

  /**
   * Helper method for calling the right binding error method, depending on whether the given value is a
   * {@link ConfigString} or not.
   */
  static Problem bindingError(Parameter parameter, Object value, List<Problem> children) {
    if (value instanceof ConfigString cs) {
      if (!cs.isAnonymous()) {
        return get().bindingErrorWithLocation(parameter, cs.getLocation(), children.toArray(size -> new Problem[size]));
      }
    }

    return get().bindingError(parameter, children.toArray(size -> new Problem[size]));
  }

  /**
   * No '<paramName>' value was provided, using default value of '<defaultValue>'
   */
  Problem applyingDefault(String paramName, Object defaultValue);

  Problem bindingError(Parameter parameter, Problem... children);

  /**
   * Alternative to just `problemsFound` that includes the location where the problematic value was defined
   */
  Problem bindingErrorWithLocation(Parameter parameter, URI location, Problem... children);

  /**
   * '<paramA>' and '<paramB>' cannot both be specified at the same time
   */
  Problem mutuallyExclusive(String paramA, String paramB);

  Problem wrongNumberGiven(String paramName, String expected, int given);

  /**
   * Return this when there are a set of optional parameters, but at least one of them is required
   */
  Problem oneOfTheseRequired(List<String> paramNames);

  /**
   * Convenience wrapper for {@link #oneOfTheseRequired(List)} (because Arrays
   * don't render very nicely in error messages).
   */
  static Problem oneOfTheseRequired(String... paramNames) {
    return get().oneOfTheseRequired(Arrays.asList(paramNames));
  }

  /**
   * Use when the parameters for a given object are invalid, e.g. produces a message like:
   * here was a problem with the parameters for <thing>
   */
  Problem invalidFor(Object thing);

  /**
   * A parameter was given, but it was ignored.
   */
  @SeverityLevel(Severity.WARNING)
  Problem ignored(String key);

  /**
   * A parameter was given, but it was ignored.
   */
  @SeverityLevel(Severity.WARNING)
  Problem ignoredWithHints(Set<String> keys, Set<String> available);

  /**
   * There was no binder available to bind `from` -> `to`.  This is not usually an error a user can do anything about,
   * but this problem is here to i18n the exception
   */
  Problem noBindingAvailable(Class<?> from, Class<?> to);

  /**
   * As above, but with a parameter for context
   */
  Problem noBindingAvailableForParameter(Parameter parameter, Class<?> from, Class<?> to);
}
