/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import lombok.Getter;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.problem.Problem;

/**
 * Thrown from BindingContext when a binding fails because no {@link ParameterBinder} is available to bind with the
 * given from and to
 *
 * Used in preference to {@link UnbindableParameterException}.  See {@link BindingContext} for info on the deprecation
 */
public class NoBindingAvailableException extends RiskscapeException {

  private static final long serialVersionUID = 3144248015313253999L;

  @Getter
  private Class<?> to;

  @Getter
  private Class<?> from;

  /**
   * A parameter associated with the {@link NoBindingAvailableException}, may be null
   */
  @Getter
  private Parameter parameter;

  private Problem withParameterVersion;

  public NoBindingAvailableException(Class<?> from, Class<?> to) {
    super(ParamProblems.get().noBindingAvailable(from, to));
    this.from = from;
    this.to = to;
  }

  /**
   * Update this exception with some parameter context info
   */
  public void setParameter(Parameter parameter) {
    this.parameter = parameter;
    this.withParameterVersion = ParamProblems.get().noBindingAvailableForParameter(parameter, from, to);
  }

  @Override
  public Problem getProblem() {
    if (withParameterVersion != null) {
      return withParameterVersion;
    } else {
      return super.getProblem();
    }
  }

}
