/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.util.Optional;

import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Root exception that all exceptions from Riskscape engine code should extend.
 * Can optionally accept a Problem, indicating that there's something the user
 * should know about this exception and maybe it's something they can fix - when
 * the exception is caught, the details of the Problem are then conveyed to the user.
 *
 * If the exception has no problem, it's likely to be an unhandled exception
 * that will need a bug fix - consider in this case whether you really should
 * use RiskscapeException, or an existing java library exception.
 *
 */
@SuppressWarnings("serial")
public class RiskscapeException extends RuntimeException {

  private final Problems problem;

  /**
   * @deprecated  Use {@link RiskscapeException#RiskscapeException(Problem)} instead to produce an internationalizable
   * error message that works with riskscape's problem API
   *
   * XXX do we want to split up the hierarchy somewhat in to : a) unexpected errors that are probably not
   * recoverable/fixable and b) exceptions that are "semantic vessels for a problem" such as a
   * NoSuchObjectExistsException - these things help us with flow control / where they serve as sort of case classes
   * where a standard {@link ResultOrProblems} isn't sufficient, e.g. we need to do something different depending on
   * the type of {@link Problem}.  More thinking required...
   */
  @Deprecated
  public RiskscapeException(String message) {
    super(message);
    this.problem = Problems.NONE;
  }

  /**
   * @deprecated Use {@link RiskscapeException#RiskscapeException(Problem, Throwable)} instead to produce an
   * internationalizable
   * error message
   * that works with riskscape's problem API
   */

  @Deprecated
  public RiskscapeException(String message, Throwable cause) {
    super(message, cause);
    this.problem = Problems.NONE;
  }

  public RiskscapeException(Problems problem) {
    this(problem, null);
  }

  public RiskscapeException(Problems problem, Throwable cause) {
    super(problem.toString(), cause);
    this.problem = problem;
  }

  /**
   * @return True if there is a problem associated with this exception
   */
  public boolean hasProblem() {
    return problem.isPresent();
  }

  /**
   * @return the problem associated with this exception, or null if there is none
   * TODO pull this thread and convert to iproblem and see what happens
   */
  public Problem getProblem() {
    return problem.toProblem().orElse(null);
  }

  @Override
  public Throwable getCause() {
    Throwable ex = super.getCause();
    if (ex == null) {
      // check if underlying problems holds an exception
      ex = problem.stream().flatMap(p -> Optional.ofNullable(p.findAnyException()).stream()).findFirst().orElse(null);
    }
    return ex;
  }

  /**
   * Returns the {@link Problems} object associated with this exception - might be NONE
   */
  public Problems getProblems() {
    return problem;
  }
}
