/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.util.Optional;
import java.util.Random;

/**
 * A global source of randomness for a riskscape process to aid reproducability of results
 */
public class RandomSingleton {

  private static Random random = new Random();
  private static Long seed;

  /**
   * @return a Random object
   */
  public static Random get() {
    return random;
  }

  public static Random get(Optional<Long> seedOr) {
    // use both the global seed (if any) and the given seedOr when seeding the new Random.
    // This means that changing the global --random-seed will reproducibly change
    // how the local seed behaves
    long globalSeed = seed == null ? 0L : seed;
    return seedOr
        .map(s -> new Random(s + globalSeed))
        // As much as possible, the seed for an RNG should itself be random.
        // Consecutive numbers make bad random seeds, so let's try to mitigate this
        // by using the seed to generate a new *random* seed.
        // See https://stackoverflow.com/a/426394
        .map(rand -> new Random(rand.nextLong()))
        .orElse(random);
  }

  /**
   * Replace the RANDOM object with a new one with the given seed.
   */
  public static void setSeed(long newSeed) {
    seed = newSeed;
    random = new Random(seed);
  }

  /**
   * @return the random seed that was last used with {@link #setSeed(long)}.  Will be null if using a default, randomly
   * seeded Random object.
   */
  public static Long getSeed() {
    return seed;
  }

}
