/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.net.URI;
import java.util.function.Consumer;

import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.config.Config;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.UserDefinedParameter;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.Bookmarks;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.geo.GeometryValidation;
import nz.org.riskscape.engine.model.IdentifiedModel;
import nz.org.riskscape.engine.pipeline.ExecutionContext;
import nz.org.riskscape.engine.pipeline.Pipeline;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;

/**
 * Contains Riskscape project resources, such as:
 * - types
 * - functions
 * - models
 * - bookmarks
 */
public interface Project extends IdentifiedLocator {

  Config getConfig();

  /**
   * {@link URI} which relative resources should be resolved against.
   * @return URI
   */
  URI getRelativeTo();

  /**
   * Set the {@link URI} which relative resources should be resolved against.
   * @param relativeTo uri to resolve relative resources against.
   */
  void setRelativeTo(URI relativeTo);

  Bookmarks getBookmarks();

  FunctionSet getFunctionSet();

  TypeSet getTypeSet();

  IdentifiedCollection<IdentifiedModel> getIdentifiedModels();

  /**
   * @return any user-defined parameters that can help customize a model for novice users
   */
  IdentifiedCollection<UserDefinedParameter> getParameterTemplates();

  URI getOutputBaseLocation();

  void setOutputBaseLocation(URI newOutputBaseLocation);

  /**
   * Validates the project for correctness, passing any error/problems encountered to the problem consumer.
   * @param problemConsumer consumer of any encountered problems.
   */
  void validate(Consumer<Problem> problemConsumer);

  /**
   * Convenience for {@code getBookmarks().add(Bookmark)}
   * @param bookmark
   */
  void add(Bookmark bookmark);

  /**
   * Convenience for {@code getFunctionSet().add(IdentifiedFunction)}
   * @param function
   */
  void add(IdentifiedFunction function);

  /**
   * @return a new {@link BindingContext} based on the state of the {@link Project}
   */
  BindingContext newBindingContext();

  /**
   * @param context reuse this realization context in the binding context
   */
  BindingContext newBindingContext(RealizationContext context);

  /**
   * @return {@link Engine} that should be used to run this project.
   */
  Engine getEngine();

  /**
   * @return {@link SRIDSet} that should be used for this project
   */
  SRIDSet getSridSet();

  /**
   * @return the default geometry validation mode to apply to bookmarks and other places where geometry
   * validation can be applied
   */
  GeometryValidation getGeometryValidation();

  void setGeometryValidation(GeometryValidation mode);

  /**
   * A default CRS to apply to geometry when none is available.  Should be used as little as possible, intended for
   * use when constructing relations when a user or data source hasn't specified a CRS.
   */
  CoordinateReferenceSystem getDefaultCrs();

  /**
   * @return a new context object for Realization.
   *
   * ## Implementation Notes
   *
   * At the moment, it's possible that a context can be reused for any project, but the intention is that a context
   * would be created and used across the realization and execution of any {@link Pipeline}.  Eventually this object
   * will be used for resource management, as well as just a convenient way of wrapping up a bunch of related bits
   */
  RealizationContext newRealizationContext();

  /**
   * @return a new {@link ExecutionContext} for this project, constructed via the engine's pipeline executor.
   */
  default ExecutionContext newExecutionContext() {
    return getEngine().getPipelineExecutor().newExecutionContext(this);
  }

  TypeBuilder getTypeBuilder();

  /**
   * A place for problems to go.  See interface on {@link ProblemSink} for usage.
   */
  ProblemSink getProblemSink();

}
