/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine;

import java.net.URI;

import org.geotools.referencing.CRS;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.i18n.Message;
import nz.org.riskscape.engine.output.Format;
import nz.org.riskscape.engine.output.PipelineOutputStore;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

public interface OutputProblems extends ProblemFactory {

  static OutputProblems get() {
    return Problems.get(OutputProblems.class);
  }

  /**
   * A fallback problem when an Output format is unable to handle a non EPSG defined CRS.
   *
   * This could be because non EPSG defined CRS's would need to be registered in an output specific
   * format and this has not been implemented.
   *
   * In practice it's unlikely that a non EPSG defined CRS would be used. Partly this is because:
   * -  {@link CRS#lookupEpsgCode(org.geotools.api.referencing.crs.CoordinateReferenceSystem, boolean) } will
   *    recognize CRS's from WKT as the correct EPSG code.
   * -  EPSG is the de facto standard
   *
   * This problem exists in the hopefully rear case that the above assumptions prove incorrect. Or
   * data has been used which does in fact use a custom defined CRS. Refer to GL729 for any potential future
   * support of non EPSG CRSs.
   *
   * @param crs the CRS that is unsupported
   */
  Problem unsupportedCrs(CoordinateReferenceSystem crs);

  /**
   * None of the configured output stores support this uri
   */
  Problem outputLocationNotSupported(URI location);

  /**
   * The user has specified that outputs should be saved in the given format, but this will be ignored.
   * @param format the format being ignored
   * @param outputStore the output store (PostGIS, GeoPackage) that is ignoring the format
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem userSpecifiedFormatIgnored(Format format, PipelineOutputStore outputStore);

  /**
   * A parent problem to use when the manifest cannot be written. The reasons should be added as child
   * problems.
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem cannotWriteManifest(PipelineOutputStore outputStore, URI location);

  /**
   * When the user has asked for checksumming of results but the output store does not support this.
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem checksumNotSupported(PipelineOutputStore outputStore);

  /**
   * When the output has no specific mapping for the given type, so will fall-back to writing it as text.
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem outputTypeAsText(String attribute, Type type, PipelineOutputStore outputStore);

  @Message("A problem occurred writing tuple ''{0}'' to {1} output")
  Problem errorWriting(String tupleContent, String format);

  @SeverityLevel(Problem.Severity.WARNING)
  Problem outputAlreadyExists(String filename);
}
