/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.TopologyException;
import org.locationtech.jts.operation.valid.TopologyValidationError;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;

/**
 * Problems relating to Geometry operations, errors, etc
 */
public interface GeometryProblems extends ProblemFactory {

  static GeometryProblems get() {
    return Problems.get(GeometryProblems.class);
  }

  /**
   * Tip for the user when a JTS {@link TopologyException} is hit.
   */
  Problem topologyExceptionTip();

  /**
   * When a {@link Geometry} cannot be re-projected from the source {@link CoordinateReferenceSystem}
   * to the target.
   *
   * This occurs when the geometry is larger than the targetCrs is.
   */
  Problem cannotReproject(Geometry geom, CoordinateReferenceSystem sourceCrs,
      CoordinateReferenceSystem targetCrs);

  /**
   * When no auto projection could be found. This is most likely to occur if the co-ordinate order
   * differs to what is stated in the CRS. So likely to be geometries from CSV and forceXY is
   * incorrect.
   *
   * Would also occur if the longitude is first co-ordinate but the first axis is not named
   * 'Longitude' for some reason.
   */
  Problem autoProjectionNotFound(double latitude, double longitude);

  /**
   * When a reprojection transform will ignore datum shifts.
   */
  @SeverityLevel(Severity.WARNING)
  Problem reprojectionIgnoringDatumShift(CoordinateReferenceSystem sourceCrs, CoordinateReferenceSystem targetCrs);

  /**
   * The data for {0} contained a mixture of CRSs, so has been reprojected to CRS {1}.
   */
  @SeverityLevel(Severity.WARNING)
  Problem crsMixtureReprojected(Object affected, CoordinateReferenceSystem target);

  /**
   * Return this to explain that a CRS name couldn't be turned in to a {@link CoordinateReferenceSystem}.  Will need
   * to be wrapped in some sort of context-giving problem, e.g. was it a function argument?  A bookmark parameter?
   */
  Problem unknownCrsCode(String crsCode);

  /**
   * When a geometry type is found but it is not {@link Referenced} e.g it is missing CRS
   */
  Problem notReferenced(Type geometryType);

  /**
   * The Well-Known Text format supplied by the user appears to be invalid.
   */
  Problem badWkt(String wktText);

  /**
   * When an invalid geometry has been detected and geometry fixing does not work.
   * @param geom    the invalid geometry
   * @param error   the error describing how geom is invalid
   * @param tupleContext  abbreviated content of the tuple
   */
  Problem invalid(Geometry geom, TopologyValidationError error, String tupleContext);

  /**
   * When an invalid geometry has been created due to re-projection and geometry fixing does not work.
   * @param sourceCrs   the CRS that the geometry was originally in
   * @param targetCrs   the CRS that original was re-projected to
   */
  Problem invalidPostReprojection(CoordinateReferenceSystem sourceCrs, CoordinateReferenceSystem targetCrs);

  /**
   * When an invalid geometry (original) has been fixed.
   * @param original  the original (and invalid) geometry
   * @param fixed     a fixed version of original
   * @param tupleContext  abbreviated content of the tuple
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem fixedInvalid(Geometry original, Geometry fixed, String tupleContext);

  /**
   * When an geometry has become invalid during re-projection, but has been fixed.
   * @param sourceCrs the CRS that the geometry was originally in
   * @param targetCrs the CRS that original was re-projected to
   * @param invalid   the invalid geometry
   * @param fixed     a fixed version of original
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem fixedInvalidPostReprojection(CoordinateReferenceSystem sourceCrs, CoordinateReferenceSystem targetCrs,
      Geometry invalid, Geometry fixed);

  Problem mismatchedCrs(CoordinateReferenceSystem lhsCrs, CoordinateReferenceSystem rhsCrs);

  /**
   * Could not perform an operation because we got a mix of geometry types (e.g.
   * lines and polygons) where we expected to have a homogenous set.
   *
   * @param geometryCollection provides more details of the problematic geometry
   */
  Problem mixedGeometryTypes(Object geometryCollection);
}
