/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.io.File;
import java.nio.file.Path;

import nz.org.riskscape.engine.i18n.Message;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.resource.ResourceProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * Factory for producing problems that involve a specific file, e.g. can't open
 * file, bad format, etc.
 *
 * @see ResourceProblems for more similar problems  - they should probably be merged and all the file problems can
 * accept URI args ( as long as we have some decent logic for displaying relative and file uris)
 */
public interface FileProblems extends ProblemFactory {

  static FileProblems get() {
    return Problems.get(FileProblems.class);
  }

  /**
   * Produces an error like: <file> is in an unsupported format
   */
  @Message("{0} is in an unsupported format")
  Problem unsupportedFormat(File file);

  /**
   * Produces an error like: Could not write to <file>
   */
  @Message("Could not write to file {0}")
  Problem cantWriteTo(Path filepath);

  /**
   * An error when a uri value is specified and backslashes are found that can't be soundly converted to forward slashes
   */
  @Message("Back-slashes - `\\` - are not allowed in URIs.  Bad URI was {0}")
  Problem backslashesNotAllowedInUris(String badUri);

  /**
   * An otherwise bad URI, typically because URI.create threw a URISyntaxException
   */
  @Message("''{0}'' is a bad URI - {1}")
  Problem badUri(String givenURI, String underlyingReason);

  /**
   * Geotools just plain can't open the file
   */
  @Message("Could not read ''{0}''. There may be something specific about how this geospatial data was saved "
    + "that RiskScape does not support, or the file may have gotten corrupted somehow")
  Problem geotoolsCannotRead(File file);

  /**
   * Tip for when we hit geospatial data that we can't open
   */
  @Message("Try opening ''{0}'' in a GIS application and check that the file is valid. "
    + "One solution may be to try re-saving the {1} data again, or try saving in a different format. "
    + "Or, in case there is an issue with the CRS metadata, "
    + "you could try using a bookmark with ''crs-name'' explicitly set")
  Problem badGeospatialDataTip(File file, String format);
}
