/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.util.List;

import com.google.common.collect.Range;

import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * Factory for producing problems that involve arguments - either to
 * a function or a pipeline step.
 */
public interface ArgsProblems extends ProblemFactory {

  Problem notNumeric(FunctionArgument argument, Type actual);

  // Wrong number of arguments provided. Expected <expected>, but got <actual>
  Problem wrongNumber(int expected, int actual);

  Problem wrongNumber(Range<Integer> expected, int actual);

  /**
   * When a variable number of arguments is allowed, but the number supplied is outside of that range.
   * @param min lest number of arguments required
   * @param max greatest number of arguments allowed
   * @param actual the number of arguments supplied
   */
  Problem wrongNumberRange(int min, int max, int actual);

  // Could not apply <target> to given arguments <givenTypes>
  Problem realizableDidNotMatch(IdentifiedFunction target, List<Type> givenTypes);

  Problem realizableDidNotMatch(Class<?> target, List<Type> givenTypes);

  /**
   * Convenient way to generate a Problem when there's a type mismatch for one of
   * the supplied arguments.
   */
  static Problem mismatch(FunctionArgument argument, Type actual) {
    return Problems.get(TypeProblems.class).mismatch(argument, argument.getType(), actual);
  }

  /**
   * Convenient way to generate a Problem when there's a type mismatch for one of
   * the supplied arguments. Used when the function allows the argument to be one of the allowedTypes
   * but actual is not one of them.
   */
  static Problem mismatch(FunctionArgument argument, Type actual, List<Type> allowedTypes) {
    return Problems.get(TypeProblems.class).requiresOneOf(argument, allowedTypes, actual);
  }

  static ArgsProblems get() {
    return Problems.get(ArgsProblems.class);
  }

  /**
   * A required keyword argument is not present
   *
   * NB this is currently only likely to happen for aggregate functions, as they realize differently.  However once we
   * address GL#114 we might support named arguments in plain function calls as well without the existing ordering
   * restriction, and so this might be a nicer/better error message than the arity error we have
   */
  Problem required(String keyword);


}
