/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.dsl;

import java.util.Collections;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.collect.Sets;

/**
 * Functional interface that replaces using {@link Pattern} for sometimes easier, more understandable and more
 * flexible pattern matching during lexing
 */
@FunctionalInterface
public interface TokenMatcher {

  /**
   * Creates a token matcher for a single character
   */
  static TokenMatcher forChar(char character) {
    return (tokenType, source, position) -> {
      if (source.charAt(position) == character) {
        return new Token(tokenType, source, position, position + 1, Character.toString(character));
      } else {
        return null;
      }
    };
  }

  /**
   * Creates a token matcher for a single character
   */
  static TokenMatcher forString(String string) {
    return (tokenType, source, position) -> {
      if (source.regionMatches(position, string, 0, string.length())) {
        return new Token(tokenType, source, position, position + string.length(), string);
      } else {
        return null;
      }
    };
  }

  /**
  * Creates a token matcher for a single character
  */
 static TokenMatcher forIString(String string) {
   return (tokenType, source, position) -> {
     if (source.regionMatches(true, position, string, 0, string.length())) {
       return new Token(tokenType, source, position, position + string.length(), string);
     } else {
       return null;
     }
   };
 }

  static TokenMatcher forPattern(String regex) {
    return forPattern(Pattern.compile(regex));
  }
  /**
   * Creates a token matcher for a {@link Pattern}
   */
  static TokenMatcher forPattern(Pattern pattern) {
    return (tokenType, source, position) -> {
      Matcher match = pattern.matcher(source.substring(position));
      if (match.find()) {
        String tokenValue;

        if (match.groupCount() == 1) {
          tokenValue = match.group(1);
        } else {
          tokenValue = match.group();
        }

        // for pattern based token matches, the source doesn't match the value - the regex can be used to 'pluck'
        // the interesting part out.  This is mostly going to be the case for things like quoted strings, although
        // we now have a purpose built matcher for that to avoid having to deal with horrific regular expressions
        return new Token(tokenType, source, position, match.end() + position, tokenValue);
      } else {
        return null;
      }
    };
  }

  Set<Character> DEFAULT_QUOTES = Sets.newHashSet('\'', '"');

  /**
   * Creates a token matcher for a quoted string.  Strings can be quoted with either single or double quotes and
   * escaping is done by adding a backslash in front of the used quote (aka the mode) character
   */
  static TokenMatcher forQuotedString() {
    return forQuotedString(DEFAULT_QUOTES);
  }

  /**
   * Creates a token matcher for a quoted string.  Strings can be quoted with either single or double quotes and
   * escaping is done by adding a backslash in front of the used quote (aka the mode) character
   */
  static TokenMatcher forQuotedString(char quoteChar) {
    return forQuotedString(Collections.singleton(quoteChar));
  }

  /**
   * Creates a token matcher for a string quoted with any of the characters in the given set.  The quote characters
   * operate in a 'mode' - that is if one of these characters is seen in the source, that triggers that 'mode' and the
   * quoting is terminated by seeing that character again.
   *
   * Escaping is always done with a backslash
   */
  static TokenMatcher forQuotedString(Set<Character> quoteWith) {
    return (tokenType, source, position) -> {
      char character = source.charAt(position);

      if (quoteWith.contains(character)) {
        final char mode = character;
        final char escapeChar = '\\';
        final StringBuilder builder = new StringBuilder();

        int curPosition = position + 1;
        boolean lastWasEscape = false;

        while (curPosition < source.length()) {
          final char curChar = source.charAt(curPosition);
          if (curChar == escapeChar) {
            // escaped escape!
            if (lastWasEscape) {
              builder.append(escapeChar);
              lastWasEscape = false;
            } else {
              lastWasEscape = true;
            }
          } else {
            if (lastWasEscape) {
              if (curChar == mode) {
                builder.append(curChar);
              } else {
                builder.append(escapeChar);
                builder.append(curChar);
              }
            } else {
              if (curChar == mode) {
                // we've hit the end of the string - we're done
                return new Token(tokenType, source, position, curPosition + 1, builder.toString());
              } else {
                builder.append(curChar);
              }

            }

            lastWasEscape = false;
          }


          curPosition++;
        }
        throw new LexerException(LexerProblems.get().eofInString(curPosition));
      } else {
        return null;
      }
    };
  }

  /**
   * @param source string to match a token from
   * @param position place in string where to start matching from
   * @return a {@link Token} that is built from `position` in `source`, or `null` if it doesn't match
   */
  Token match(TokenType type, String source, int position);

  String[] LINE_ENDINGS = new String[] {"\r\n", "\r", "\n"};

  static TokenMatcher forLineComment(String startsWith) {
    return (TokenType type, String source, int position) -> {
      // do a cheap check first before doing a complete more expensive check after
      if (source.charAt(position) == startsWith.charAt(0)
          && source.substring(position, position + startsWith.length()).equals(startsWith)
      ) {
        int i = position;

        lineEndScanning:
        for (; i < source.length(); i++) {
          for (int j = 0; j < LINE_ENDINGS.length; j++) {
            String lineEnding = LINE_ENDINGS[j];

            // again - do a cheap check first
            if (source.charAt(i) == lineEnding.charAt(0)
                && source.substring(i, i + lineEnding.length()).equals(lineEnding)) {
              break lineEndScanning;
            }
          }
        }
        return new Token(type, source, position, i, source.substring(position + startsWith.length(), i));
      } else {
        return null;
      }
    };
  }

  default Optional<String> getKeyword() {
    return Optional.empty();
  }
}
