/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.dsl;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import nz.org.riskscape.dsl.Lexer.Tokens;

/**
 * A substring, location in the source text, plus type, that constitutes a {@link Token} emitted from the
 * {@link Lexer}
 */
// source code metadata is excluded from equality on purpose
@EqualsAndHashCode(of = {"type", "value"})
public class Token {

  public static final Token UNKNOWN_LOCATION = Token.token(null, "");

  public static Token eof(Tokens<?> tokens, String source) {
    return new Token(tokens.eofToken, source, source.length(), source.length(), "");
  }

  /**
   * Handy constructor for testing situations
   */
  public static Token token(TokenType tokenType, String value) {
    return new Token(tokenType, value, 0, value.length(), value);
  }

  @Getter
  public final TokenType type;
  public final String source;
  public final int begin;
  public final int end;

  @Getter(AccessLevel.PACKAGE)
  private Token previous;

  @Getter(AccessLevel.PACKAGE)
  private Token next;

  @Getter
  public final String value;

  /*
   * This isn't final, but should be treated as though it is immutable - ideally we would accept the SourceLocation here
   * (instead of begin) but similar to LexerProblems#eofInString, it's too much change right now to do this.
   */
  @Getter @Setter
  private SourceLocation location;

  public Token(TokenType type, String source, int begin, int end, String value) {
    this.type = type;
    this.begin = begin;
    this.end = end;
    this.source = source;
    this.value = value;
    // avoid NPEs
    this.location = SourceLocation.unlined(begin);
  }

  public Token getStartOfLine() {
    if (previous == null) {
      return this;
    } else {
      return previous.getStartOfLine();
    }
  }

  /**
  * @return a List of Token objects that are the whitespace tokens that precede any non-whitespace tokens.
   */
  public List<Token> getIndents() {
    Token ptr = getStartOfLine();
    List<Token> beingBuilt = new LinkedList<>();
    while (ptr != null && ptr.isWhitespace()) {
      beingBuilt.add(ptr);
      ptr = ptr.next;
    }

    return beingBuilt;
  }

  /**
   * @return The number of whitespace tokens that precede any non-whitespace tokens.
   */
  public int countIndents() {
    Token ptr = getStartOfLine();
    int counter = 0;
    while (ptr != null && ptr.isWhitespace()) {
      counter++;
      ptr = ptr.next;
    }

    return counter;
  }

  /**
   * @return true if this token is of a whitespace token type
   */
  public boolean isWhitespace() {
    return type.isWhitespace();
  }

  @Override
  public String toString() {
    return "['" + value + "' type=" + type+ " (" + location + ")]";
  }

  public String getPosition() {
    return begin + "..." + end;
  }

  public String rawValue() {
    return source.substring(begin, end);
  }

  /**
   * @return true if this token has more indent tokens than the previous one
   */
  public boolean moreIndented(Token compareTo) {
    Iterator<Token> myIndents = this.getIndents().iterator();
    Iterator<Token> theirIndents = compareTo.getIndents().iterator();

    int myCount = 0;
    int theirCount = 0;

    Token none = Token.token(compareTo.type, "");

    while (myIndents.hasNext() || theirIndents.hasNext()) {
      Token mine = myIndents.hasNext() ? myIndents.next() : none;
      Token theirs = theirIndents.hasNext() ? theirIndents.next() : none;

      if (mine != none && theirs != none && mine.type != theirs.type) {
        throw new InconsistentIndentException(mine, theirs);
      }

      myCount += mine.value.length();
      theirCount += theirs.value.length();
    }

    return myCount > theirCount;
  }


  // package level on purpose - set during lexing by the lexer to allow indent tracking
  // implementation note - making the tokens themselves a linked list seemed a good compromise - the alternative
  // that would have also allowed arbitrary rewinding, would have meant keeping a list of indents (or counts) for
  // each line that had been parsed so far.  This implementation is slightly simpler in that all the lexer has to
  // do is update pointers as it lexes and delegate logic to it.  Given the ASTs already point back to their tokens, the
  // only overhead we add here is an extra pair of pointers for each token.
  void setPrevious(Token toSet) {
    this.previous = toSet;
    if (toSet != null) {
      toSet.next = this;
    }
  }

  /**
   * @return true if the source code's non-whitespace tokens occur on a single line
   */
  public boolean isSourceSingleLine() {
    return source.trim().indexOf("\n") == -1;
  }
}
