/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.dsl;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import lombok.Data;

/**
 * Used to represent a parsing operation that's conditional on a sequence of token possibilities.
 *
 * This type is used by {@link Lexer#tryThese(ConditionalParse...)} to help parses model a list of precedence ordered
 * lookahead rules that give the basis of a grammar.
 * @param <T> the type of thing returned by a parsing operation
 */
@Data
public class ConditionalParse<T extends Object> {

  /**
   * Construct a {@link ConditionalParse} that matches a sequence of possible tokens.
   *
   * Given the list of sets `{S1, S2, S3}`, then the
   * returned {@link ConditionalParse} object would apply to a sequence of tokens where the next three tokens types
   * are first in `S1`, second in `S2` and then thirdly from `S3`.  If this list was `[{a, b}, {c}, {d, e}]`, then it
   * would apply to `[a, c, d]`, `[b, c, d]`, `[a, c, e]` and `[b, c, e]`.
   *
   * @param name debug identifier for the parsing step
   * @param nextTokens the sequence of possible token types to match.
   * @param callback a function to call that returns the results of parsing - it's assumed that the lexer is provided
   * by the function's enclosing scope.
   */
  public static <T extends Object>
  ConditionalParse<T> parseIf(String name, List<Set<? extends TokenType>> nextTokens, Supplier<T> callback) {
    return new ConditionalParse<>(name, nextTokens, callback);
  }

  /**
   * Convenience version of {@link #parseIf(String, List, Supplier)} with a singleton list.
   */
  public static <T extends Object>
  ConditionalParse<T> parseIfOneOf(String name, Set<? extends TokenType> nextTokens, Supplier<T> callback) {
    return new ConditionalParse<>(name, Collections.singletonList(nextTokens), callback);
  }

  /**
   * Convenience version of parseIf where the next token has to be exactly the given token.
   */
  public static <T extends Object>
  ConditionalParse<T> parseIfIs(String name, TokenType nextToken, Supplier<T> callback) {
    return new ConditionalParse<>(name, Collections.singletonList(Collections.singleton(nextToken)), callback);
  }

  /**
   * Convenience version of parseIf where the next sequence of tokens needs to match the sequence of token types in
   * `nextTokens`
   */
  public static <T extends Object>
  ConditionalParse<T> parseIfIs(String name, List<? extends TokenType> nextTokens, Supplier<T> callback) {
    List<Set<? extends TokenType>> list = nextTokens.stream()
        .map(nt -> Collections.singleton(nt))
        .collect(Collectors.toList());

    return new ConditionalParse<>(name, list, callback);
  }

  /**
   * Debugging name for this part of the grammar
   */
  private final String name;

  /**
   * The sequence of tokens to be matched
   */
  private final List<Set<? extends TokenType>> nextTokens;

  /**
   * A callback that gives the results of parsing.
   */
  private final Supplier<T> parse;

}
