/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config;

import java.net.URI;
import java.util.stream.IntStream;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.resource.Resource;

/**
 * A {@link ConfigString} is a wrapper around some other string that includes the location of this value, i.e. the file
 * resource that defines it (like an ini file).
 *
 * This class can be used with binding to transparently include the locations of problematic values, so that error
 * messages can direct the user to the problem.
 */
public interface ConfigString extends CharSequence {

  /**
   * Utility base class.
   */
  @RequiredArgsConstructor
  abstract class Wrapper implements ConfigString {

    private final CharSequence value;

    @Override
    public CharSequence subSequence(int start, int end) {
      return value.subSequence(start, end);
    }

    @Override
    public int length() {
      return value.length();
    }

    @Override
    public char charAt(int index) {
      return value.charAt(index);
    }

    @Override
    public IntStream chars() {
      return value.chars();
    }

    @Override
    public IntStream codePoints() {
      return value.codePoints();
    }

    @Override
    public boolean equals(Object obj) {
      if (obj instanceof CharSequence rhs) {
        return rhs.equals(value);
      }
      return false;
    }

    @Override
    public String toString() {
      return value.toString();
    }

    @Override
    public int hashCode() {
      return value.hashCode();
    }
  }

  /**
   * @return a ConfigString that wraps the given value, but without a known location.
   */
  static ConfigString anon(CharSequence value) {
    return wrap(Resource.UNKNOWN_URI, value);
  }

  /**
   * @return a {@link ConfigString} that uses the given location and value
   */
  static ConfigString wrap(URI location, CharSequence value) {
    return new Wrapper(value) {
      @Override
      public URI getLocation() {
        return location;
      }
    };
  }

  /**
   * The string being wrapped
   */
  @Override
  String toString();

  /**
   * @return a location where this {@link ConfigString} was defined.
   */
  URI getLocation();

  /**
   * @return true if this {@link ConfigString}s location is {@link Resource#UNKNOWN_URI}
   */
  default boolean isAnonymous() {
    return getLocation().equals(Resource.UNKNOWN_URI);
  }
}
