/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cli;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import nz.org.riskscape.picocli.CommandLine.Help.Ansi;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.IStyle;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.Text;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;

public interface Terminal extends ProblemSink {

  /**
   * Convenient constant to denote no styling
   */
  List<IStyle> NO_STYLES = Collections.emptyList();

  /**
   * Convenient constant to use in place of null for empty, un-styled, text
   */
  Text EMPTY_TEXT = Ansi.OFF.apply("", NO_STYLES);

  /**
   * Ellipsis character `...`
   */
  char ELLIPSIS = '\u2026';

  InputStream getIn();

  PrintStream getOut();

  /**
   * @return an {@link AnsiPrintStream} to use, wrapping stdout.
   */
  AnsiPrintStream getAnsiOut();

  PrintStream getErr();

  /**
   * @return an {@link AnsiPrintStream} to use, wrapping stderr.
   */
  AnsiPrintStream getAnsiErr();

  /**
   * Prints a line of text to the console with a new line after, returning an ansi print stream in case you want to
   * print more
   */
  default AnsiPrintStream println(Object message) {
    return getAnsiOut().println(message);
  }

  /**
   * Prints an array of problems using the given terminal's problem logging mechanism.  Note that this is likely to send
   * the text to stderr and not stdout.
   */
  default void printProblems(Problem... problems) {
    for (int i = 0; i < problems.length; i++) {
      log(problems[i]);
    }
  }

  /**
   * Prints an array of problems using the given terminal's problem logging mechanism.  Note that this is likely to send
   * the text to stderr and not stdout.
   */
  default void printProblems(List<Problem> problems) {
    problems.forEach(this::log);
  }

  /**
   * Update the terminal's status area with the following text.  This should be ignored if isTTY false.  May also be
   * ignored if the terminal implementation doesn't support it.
   * @param lines status lines to present to user. If null or empty any previous status should be removed.
   */
  default void updateStatus(List<String> lines) {
    //  noop
  }

  /**
   * @return the locale that the terminal is in - this should influence what text you send to it
   */
  Locale getLocale();

  /**
   * @return a single line of text from the user
   * @throws IOException if the underlying stream threw an exception
   * @throws RuntimeException if the underlying implementation supports it, this can also throw an exception if the
   * input is interrupted, say, by ctrl-c
   */
  String readline() throws IOException;

  /**
   * @return true if this terminal is a tty or tty emulated.  Will be false if it looks like RiskScape is being used
   * as part of a unix pipeline e.g. `riskscape | less`
   */
  boolean isTTY();

  /**
   * @return the width (in standard character-width) of the user's terminal window.
   */
  int getDisplayWidth();

  /**
   * @return the height (in standard character-height) of the user's terminal window.
   */
  int getDisplayHeight();

  /**
   * @return an Ansi object to use for styling text - will be whatever this terminal can support or is configured to
   * support
   */
  Ansi getAnsi();

  /**
   * Specify an interrupt handler that will be called on ctrl-c when readline isn't in progress.  Should default to
   * exiting.
   * @return the previous handler.
   */
  InterruptHandler setFallbackHandler(InterruptHandler newFallbackHandler);

}
