/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cli;

import java.io.PrintStream;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.IStyle;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.Style;

/**
 * A more convenient print stream API to use for printing things out to the console, supporting ANSI styling if the
 * underlying terminal supports it and fluid use (e.g. it returns itself)
 */
@RequiredArgsConstructor
public final class AnsiPrintStream implements AnsiHelper {

  @Getter
  private final Ansi ansi;

  private final PrintStream out;

  /**
   * Create a print stream wrapping the given terminal's stdout
   */
  public AnsiPrintStream(Terminal terminal) {
    this.ansi = terminal.getAnsi();
    this.out = terminal.getOut();
  }

  /**
   * Print the given object's toString representation raw (no ansi escaping) to the underlying stream.
   */
  public AnsiPrintStream print(Object object) {
    out.print(object);
    return this;
  }

  /**
   * Print the given object's toString representation raw (no ansi escaping) to the underlying stream.  A new line is
   * appended.
   */
  public AnsiPrintStream println(Object object) {
    return print(object).println();
  }

  /**
   * Print the given text to the underlying stream, wrapping it in the given styles (if supported)
   */
  public AnsiPrintStream printStyles(String plainText, IStyle... styles) {
    out.print(applyStyles(plainText, styles));
    return this;
  }

  /**
   * Print the given text to the underlying stream, wrapping it in the given styles (if supported).  A new line is
   * appended.
   */
  public AnsiPrintStream printlnStyles(String plainText, Style... styles) {
    return printStyles(plainText, styles).println();
  }

  /**
   * Print the given text to the underlying stream, parsing picocli markup and converting it to have the correct escape
   * codes
   */
  public AnsiPrintStream printMarkup(String markupText) {
    out.print(applyMarkup(markupText));
    return this;
  }

  /**
   * Print the given text to the underlying stream, parsing picocli markup and converting it to have the correct escape
   * codes. A new line is appended.
   */
  public AnsiPrintStream printlnMarkup(String markupText) {
    return printMarkup(markupText).println();
  }

  /**
   * Print a new line to the underlying stream
   */
  public AnsiPrintStream println() {
    out.println();
    return this;
  }

}
