# Old getting started guide

All asset and hazard data has been prepared by the RiskScape development team and is freely distributed.
Click [Here](../getting-started.zip) to download a zip file containing the getting-started example files.
If you are running the RiskScape docker image, then these files will already exist under
`/opt/riskscape-cli/examples/getting-started/`.

In our example the assets will be buildings. The attributes for each building are:

* Location (geometry)
* Name
* Construction (either timber, brick, stone, steel or concrete)
* Number of Stories (how tall it is)

The hazard is an attack from a Kaijū . The things that we know about a Kaijū attack are:

* Location (geometry)
* Name (of the Kaijū that attacked)
* Severity (of the attack)

.. warning::

  The affect of Kaijū attacks upon any building, as described in this tutorial
  has no scientific basis.



## Applying RiskScape concepts to your model

Let's briefly recap how the concepts in the RiskScape :ref:`overview` apply to the Kaijū attack we will model.

* :ref:`functions` take a given asset (a building) and a hazard (a Kaijū attack) and calculate an outcome (the resulting building damage).
* :ref:`models` take *all* your assets and *all* your hazard information, and apply your function to each one individually.
Models can also specify additional processing, such as aggregating the results in different ways.
* :ref:`types` ensure that your model's source data has all the information (attributes) that the function needs
to calculate an accurate outcome.
Our Kaijū model's function will need a `building` type (with `construction` and `stories` attributes),
and an `kaiju_attack` type (with a `severity` attribute).
* :ref:`Bookmarks <bookmarks>` define the source files for our asset and hazard data (i.e. CSV file, shapefile, etc),
and can define how we map the raw data to a RiskScape type.
* :ref:`projects` tell RiskScape about all the functions, types, bookmarks, and model details it needs to use.

## Setup your project directory

:ref:`projects` are a way to organize your RiskScape files,
i.e. you can use separate projects when you need to model different things.
We recommend using a separate directory for each RiskScape project,
although you don't have to.

For Windows users, it's simplest to specify your RiskScape project file
as part of a :ref:`Desktop shortcut <shortcut_script>`. You may have already setup a shortcut
as part of the installation.

If you're not using the Windows shortcut, then do the following:
* Create a directory for this project, e.g. `C:\RiskScape-Projects\getting-started`.
* In your terminal prompt, `cd` to your project directory.
* Create a blank text file in this directory called `project.ini`.
We'll add to this file throughout the example.
* In your terminal, set a `RISKSCAPE_PROJECT` environment variable with the full file path
of the project file. :ref:`specify_project` has more details on how to do this.

In a Windows Command Prompt, the above would look like:
```text
mkdir C:\RiskScape_Projects\getting-started
cd C:\RiskScape_Projects\getting-started
echo. >> project.ini
set RISKSCAPE_PROJECT=C:\RiskScape_Projects\getting-started\project.ini
```
.. note::

  We use ``RISKSCAPE_PROJECT`` here to simplify the commands in this example.
  Alternatively, you could specify the ``--project`` argument each time RiskScape is run,
  e.g. ``riskscape --project=./project.ini``.

## Define types

RiskScape uses :ref:`types` to set what data is expected as both inputs and outputs.

Our example will have the following types:

* `building` the asset
* `kaiju_attack` the hazard
* `building_attack_outcome` the result

When constructing the types to it is good to consider what:

* attributes are required to calculate a loss
* attributes may be required to sort/aggregate output
* the source data has available.

Open your `project.ini` file in an text editor (e.g. Notepad) and add the following:
```
[type kaiju_attack]
type.geom = geometry
type.kaiju_name = text
type.severity = lookup('kaiju_attack_severity')

[type kaiju_attack_severity]
type = range(floating, 0, 10)

[type building]
type.geom = geometry
type.name = text
type.stories = integer
type.construction = enum('timber', 'concrete', 'steel','stone','brick')

[type building_attack_outcome]
type.damage = integer
```
The `kaiju_attack` type identifies which Kaijū was involved in a given attack (`kaiju_name`)
and where the attack occurred (`geom`). It also stores the `severity` of the
attack as `kaiju_attack_severity`, which is another type we have defined.

The `kaiju_attack_severity` type is a floating-point number that we expect to be
within a given range. This makes the severity of
the attack more meaningful, where 0 is nothing and 10 is a complete hulk smash.

The `building` type allows us to define the types of building construction we are interested in.

.. tip::
  When naming things in RiskScape, we recommend using words in lower-case characters
  separated by underscores for the IDs, i.e. ``kaiju_attack`` rather than ``kaiju-attack``.

  Although IDs can contain characters like ``-`` or ``:``, it means you need to surround the ID
  in double-quotes whenever it gets used in an expression, e.g. ``"kaiju-attack"``.

  This applies to all IDs used to name types, functions, bookmarks and models.

To list types available to RiskScape:
```none
riskscape type list
```

In your list you should see the types you have added.
```none
+------------------------+-----------------------------------------------------+
|id                      |description                                          |
+------------------------+-----------------------------------------------------+
|building_attack_outcome |{damage=>Integer}                                    |
|                        |                                                     |
|kaiju_attack            |{geom=>Geom, kaiju_name=>Text, severity=>kaiju_attack|
|                        |_severity}                                           |
|                        |                                                     |
|building                |{geom=>Geom, name=>Text, stories=>Integer,           |
|                        |construction=>Enumeration(values=[timber, concrete,  |
|                        |steel, stone, brick])}                               |
|                        |                                                     |
|kaiju_attack_severity   |WithinRange(type=Floating, min=0.0, max=10.0)        |
+------------------------+-----------------------------------------------------+
```

For more details on the built-in RiskScape types we have used here, such as
`enum()` or `range()`, refer to the `riskscape type-registry list` output.

## Create a RiskScape function

A RiskScape Function accepts some inputs (typically asset and hazard) and calculates
an outcome.

In our function we will calculate building damage as a result of a Kaijū attack.
We believe that the resulting damage will be based on the construction type, the number
of stories, and the severity of any attack.

For our first hazard, the Kaijū stomp, we are going to use a *classifier* function.

Create a file in your project directory called `kaiju-building-stomp.txt`:
```
id: kaiju_stomp
description: Models damage from a Kaiju stomping on a building

argument-types:
  building: lookup('building')
  stomp: nullable(lookup('kaiju_attack'))

pre:
  severity: if(is_null(stomp), 0.0, stomp.severity)

filter: building.construction = 'timber'
  function:
    resilience: 1

filter: building.construction = 'concrete'
  filter: building.stories > 18
    function:
      resilience: 5
  default:
    function:
      resilience: 7

filter: building.construction = 'steel'
  filter: building.stories > 80
    function:
      resilience: 3
  filter: building.stories > 55
    function:
      resilience: 5
  default:
    function:
      resilience: 9

default:
  function:
    resilience: 4

post:
  damage: (10 - resilience) * severity
```

Then add the following to your `project.ini` file:

```ini
[project]
functions = kaiju-building-stomp.txt
```

.. note::
  If your function file is located in another directory, then you would need to specify
  either the full or relative file path here.

To list the user-defined functions available to RiskScape:
```none
riskscape function list
```
Your function list should look like this
```none
+------------------+------------------+------------------------+------------------+-----------+
|id                |description       |arguments               |returnType        |category   |
+------------------+------------------+------------------------+------------------+-----------+
|kaiju_stomp       |Models damage from|[building: building,    |building_attack_  |UNASSIGNED |
|                  |a Kaijū stomping  |stomp: Nullable[kaiju_  |outcome           |           |
|                  |on a building     |attack]]                |                  |           |
|                  |                  |                        |                  |           |
+------------------+------------------+------------------------+------------------+-----------+
```

RiskScape also comes with many built-in functions you may find useful, such as `is_exposed()`,
The built-in functions are organized by categories, such as *maths* or *geometry_processing*. You can view the built-in functions available in RiskScape
by category, or use `--all` to list all of them. For example:

```none
riskscape function list --all
riskscape function list --category maths
```

.. tip::
  Some RiskScape commands output a lot of text.
  You can use ``more`` to scroll through the output, either a line-by-line
  (using the *Enter* key) or a page at a time (using *space bar*) E.g.
  ``riskscape function list --all | more``

### Nullable hazards

In the above function the `stomp` hazard type is :ref:`nullable <type-nullable>`. This is because
the :ref:`models-default` will by default call the function for all
assets regardless of there being a hazard. In other words, there will be some
New York buildings that the Kaijū doesn't attack - these buildings will still
be passed to our `kaiju_stomp` function, but the `stomp` hazard will be null.

This default behaviour can be changed by using a different `join-type` in the model.

Because of this it is advisable that functions are able to handle a null hazard.

## Data sources

Before continuing some sample data will be needed.

Create a `data` sub-directory in your RiskScape project directory.
Extract or copy the [getting-started.zip](../getting-started.zip) `data` files
into the directory you just created.
Among these files are the following data sources:
 * `new-york.csv` - A base map of New York, using Well Known Text (WKT) for the geometry.
 * `nyc-landmarks.csv` - This is our asset data, buildings with geometry, name, construction type and number of stories.
 * `kaiju-route.csv` - This is the path of our Kaijū through New York, including the severity of the attack for different sections of the path.
 * `kaiju-fire.shp` - Used in our multi-hazard analysis later, this is a shape file
 that describes the length and intensity of a number of fire blast attacks from the Kaijū.

### Bookmarking data

Bookmarks allow data sources to be accessed by a short bookmark ID.

If you don't create a bookmark, you will need to use the file path in your RiskScape command.

Bookmarks also allow for source data to be translated to :ref:`types` in various
ways. In the example below we use this to set the geometry CRS as this is not
typically available in a CSV file, and to combine the lat/long columns into a
single geometry type.

Add the following to your `project.ini` file:
```
[bookmark nyc_buildings]
description = A set of NYC landmark buildings
location = data/nyc-landmarks.csv
type = building
crs-name = EPSG:4326
crs-longitude-first = true
map-attribute.geom = geom_from_wkt(WKT)

[bookmark kaiju_route]
description = The route of our kaiju through NYC
location = data/kaiju-route.csv
type = kaiju_attack
crs-name = EPSG:4326
crs-longitude-first = true
map-attribute.geom = geom_from_wkt(WKT)
```

Note that the [EPSG:4326](https://epsg.io/4326) CRS specifies a *lat, long* axis order,
whereas our CSV data is actually in a *long, lat* format.
This means we need to set the `crs-longitude-first` parameter to tell RiskScape to
read the data as *long, lat* coordinates.
Disagreement over the latitude/longitude order is actually a
fairly common GIS problem, covered in more detail in :ref:`crs-lat-long`.

To list available bookmarks:
```none
riskscape bookmark list
```
## Constructing a model

A RiskScape model includes the model template, types, functions and
data (assets and hazards) that are used to produce a result.

### Define the model

Add the following to your `project.ini` file:
```
[model kaiju_stomp]
template = default
function = kaiju_stomp
asset = nyc_buildings
hazard = kaiju_route
geometry-join = intersects
include-hazard-in-output = true
include-asset-in-output = true

```
When the model runs, it will apply the *function* `kaiju_stomp` to the *asset* `nyc_buildings` and the *hazard* `kaiju_route`. The default for `geometry-join` is `centroid`, whereas for our model we want to use `intersects`.

To help us track what is happening, we are also including the asset and hazard data in the output file.

### Run the model

To run the model and output the results in various ways:
```none
# run with default options. This will save the output in a shapefile (because there is spatial data).
riskscape model run kaiju_stomp

# save the results in another directory
riskscape model run kaiju_stomp --output stomp-results/

# save results as csv
riskscape model run kaiju_stomp -f csv
```

Run `riskscape model run kaiju_stomp -f csv` then open the resulting file in a spread sheet program
(such as MS Excel). It should look something like the following:

```none
+---------+---------+---------+---------+---------+---------+--------+--------+--------+
|the_geom |output.d…|asset.ge…|asset.na…|asset.st…|asset.co…|hazard.…|hazard.…|hazard.…|
+---------+---------+---------+---------+---------+---------+--------+--------+--------+
|Polygon… |0        |Polygon… |Central  |1        |brick    |        |        |        |
|         |         |         |Perk     |         |         |        |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |0        |Polygon… |The      |5        |stone    |        |        |        |
|         |         |         |Hayden   |         |         |        |        |        |
|         |         |         |Planetar…|         |         |        |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |0        |Polygon… |That Cafe|1        |stone    |        |        |        |
|         |         |         |From     |         |         |        |        |        |
|         |         |         |Seinfeld |         |         |        |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |0        |Polygon… |The Met  |15       |steel    |        |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |42       |Polygon… |Statue of|22       |stone    |LineStr…|Gauwdzu…|7.0     |
|         |         |         |Liberty  |         |         |…       |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |0        |Polygon… |Natural  |6        |stone    |        |        |        |
|         |         |         |History  |         |         |        |        |        |
|         |         |         |Museum   |         |         |        |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |0        |Polygon… |Ed       |15       |brick    |        |        |        |
|         |         |         |Sullivan |         |         |        |        |        |
|         |         |         |Theatre  |         |         |        |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |24       |Polygon… |NYSE     |8        |stone    |LineStr…|Gauwdzu…|4.0     |
|         |         |         |         |         |         |…       |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |24       |Polygon… |Tiffany  |8        |brick    |LineStr…|Gauwdzu…|4.0     |
|         |         |         |and Co   |         |         |…       |        |        |
|         |         |         |         |         |         |        |        |        |
|Polygon… |6        |Polygon… |Federal  |16       |concrete |LineStr…|Gauwdzu…|2.0     |
|         |         |         |Reserve  |         |         |…       |        |        |
+---------+---------+---------+---------+---------+---------+--------+--------+--------+
```

.. note::

  To see what parameters may be set on your model run:
  ``riskscape model list``

  Parameters may either be included with the model definition in your ``project.ini`` file
  or specified on the command line.

## Multi-hazard analysis using a Python function

Our Kaijū is a hazard not only because it stomps on buildings, but it can also unleash
a fire blast attack. Included in the sample data bundle is the `kaiju-fire.shp` shapefile that we will
use as our hazard.

### Update the bookmark and transform the data

For each fire blast attack, our hazard data records a time and an intensity.

Our damage function require a severity that we can use. So that we can keep the source
data pure, we can add this transformation in our bookmark

Add the following to your `project.ini` file:

```
[bookmark kaiju_fire]
description = The intensity and extent of the fire blast attacks
location = data/kaiju-fire.shp
type = kaiju_attack
map-attribute.kaiju_name = name
map-attribute.severity = intensity * time
map-attribute.geom = the_geom
```

Now when we use this dataset, the severity will be calculated from the underlying source data for intensity and time.

### Write the Python function

For this model, we want to know:

* What is the damage from a stomp
* What is the damage from a scorch
* What is the combined damage to the building.
* What buildings are undamaged (damage is zero).

Instead of a classifier function, this time we will use a :ref:`Python function <python-functions>`.
Create a `kaiju-building-stomp-scorch.py` file in your project directory and paste the following:

```python
from nz.org.riskscape.engine.types import Nullable

ID = 'kaiju_stomp_scorch'
DESCRIPTION = 'Models damage from a Kaiju stomping and/or scorching a building'

ARGUMENT_TYPES = ['building', Nullable.of(typeset.getLinkedType('kaiju_attack')), \
 Nullable.of(typeset.getLinkedType('kaiju_attack'))]

RETURN_TYPE = 'building_attack_outcome'

def function(building, stomp, scorch):
    if stomp is None and scorch is None:
        # both hazards are null, so building is unaffected
        return {'damage': 0}

    # get the severity of the hazards (if a hazard is null, use severity=1
    # in the multiplication, i.e. just use the other hazard's damage)
    if stomp is None:
        stomp_severity = 1
    else:
        stomp_severity = stomp.get('severity')

    if scorch is None:
        scorch_severity = 1
    else:
        scorch_severity = scorch.get('severity')

    if building.get('construction') == 'timber':
        resilience = 1
    elif building.get('construction') == 'concrete':
        resilience = 5
    elif building.get('construction') == 'steel':
        resilience = 5
    else:
        resilience = 4

    damage = (10 - resilience) * stomp_severity * scorch_severity
    if damage > 100:
        damage = 100
    return {'damage': damage}
```

Then add the following to your `project.ini` file:
```
[project]
functions = kaiju-building-stomp-scorch.py
```

.. tip::
  We created a second ``[project]`` section here for simplicity.
  In general, it's best to have a single ``[project]`` section and
  add the new ``functions`` line there.

Confirm your new function is valid by running `riskscape function list`:

```none
+------------------+------------------+------------------------+------------------+-----------+
|id                |description       |arguments               |returnType        |category   |
+------------------+------------------+------------------------+------------------+-----------+
|kaiju_stomp       |Models damage from|[building: building,    |building_attack_  |UNASSIGNED |
|                  |a Kaijū stomping  |stomp: Nullable[kaiju_  |outcome           |           |
|                  |on a building     |attack]]                |                  |           |
|                  |                  |                        |                  |           |
|kaiju_stomp_scorch|Models damage from|[building,              |building_attack:  |UNASSIGNED |
|                  |a Kaiju stomping  |Nullable[kaiju_attack], |outcome           |           |
|                  |and/or scorching a|Nullable[kaiju_attack]] |                  |           |
|                  |building          |                        |                  |           |
+------------------+------------------+------------------------+------------------+-----------+
```

### Define the multi-hazard model

We now want to add a new model that uses the same asset, the two hazard sources for stomp and scorch, and our new damage function.

Add the following to your `project.ini`:

```
[model kaiju_stomp_scorch]
template = default
function = kaiju_stomp_scorch
asset = nyc_buildings
hazard = kaiju_route
hazard = kaiju_fire
geometry-join = intersects
include-asset-in-output = true
include-hazard-in-output = true
```
### Run the multi-hazard model

Run `riskscape model run kaiju_stomp_scorch -f csv` then open the resulting file in a spread sheet program
(such as MS Excel). It should look something like the following:

```none
+---------+---------+---------+---------+---------+---------+---------+---------+---------+---------+---------+----------+
|the_geom |output.d…|asset.ge…|asset.na…|asset.st…|asset.co…|hazard1. |hazard1. |hazard1. |hazard2. |hazard2. |hazard2.  |
|         |         |         |         |         |         |geom     |kaiju_na…|severity |geom     |kaiju_na…|severity  |
+---------+---------+---------+---------+---------+---------+---------+---------+---------+---------+---------+----------+
|Polygon… |0        |Polygon… |Central  |1        |brick    |         |         |         |         |         |          |
|         |         |         |Perk     |         |         |         |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |0        |Polygon… |The      |5        |stone    |         |         |         |         |         |          |
|         |         |         |Hayden   |         |         |         |         |         |         |         |          |
|         |         |         |Planetar…|         |         |         |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |0        |Polygon… |That Cafe|1        |stone    |         |         |         |         |         |          |
|         |         |         |From     |         |         |         |         |         |         |         |          |
|         |         |         |Seinfeld |         |         |         |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |0        |Polygon… |The Met  |15       |steel    |         |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |100      |Polygon… |Statue of|22       |stone    |LineStri…|Gauwdzul…|7.0      |MultiPol…|Gauwdzul…|4.0       |
|         |         |         |Liberty  |         |         |…        |         |         |…        |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |24       |Polygon… |Natural  |6        |stone    |         |         |         |MultiPol…|Gauwdzul…|4.0       |
|         |         |         |History  |         |         |         |         |         |…        |         |          |
|         |         |         |Museum   |         |         |         |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |0        |Polygon… |Ed       |15       |brick    |         |         |         |         |         |          |
|         |         |         |Sullivan |         |         |         |         |         |         |         |          |
|         |         |         |Theatre  |         |         |         |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |24       |Polygon… |NYSE     |8        |stone    |LineStri…|Gauwdzul…|4.0      |         |         |          |
|         |         |         |         |         |         |…        |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |24       |Polygon… |Tiffany  |8        |brick    |LineStri…|Gauwdzul…|4.0      |         |         |          |
|         |         |         |and Co   |         |         |…        |         |         |         |         |          |
|         |         |         |         |         |         |         |         |         |         |         |          |
|Polygon… |10       |Polygon… |Federal  |16       |concrete |LineStri…|Gauwdzul…|2.0      |         |         |          |
|         |         |         |Reserve  |         |         |…        |         |         |         |         |          |
+---------+---------+---------+---------+---------+---------+---------+---------+---------+---------+---------+----------+
```


## Aggregating results to a spatial layer

Included in the file bundle for the Kaijū example is a basic map of the New York
boroughs as a CSV file.

To aggregate the results to the shapes in this map, we need to first add a new
bookmark and type so that RiskScape knows what to do with this data.

Add the following to your `project.ini`:

```
[type boroughs]
type.geom = geometry
type.name = text

[bookmark nyc_boroughs]
description = A map showing the outlines of the NYC boroughs and islands
location = data/new-york.csv
type = boroughs
crs-name = EPSG:4326
crs-longitude-first = true
map-attribute.geom = geom_from_wkt(WKT)
```

You can run `riskscape type list` and `riskscape bookmark list` to confirm your changes.

Now that RiskScape knows about our map, we can use `aggregation-layer` to join
our data to this layer. For example, we can include the borough where the
building is located in our output by using:

Run `riskscape model run kaiju_stomp -p aggregation-layer=nyc_boroughs -p include-aggregation-layer-in-output=true -f csv`
then open the resulting file in a spread sheet program (such as MS Excel). It should look something
like the following:

```none
+---------+-------------+-----------+-------------+-------------+-------------+------------+-------------+------------+--------+------------+
|the_geom |output.damage|asset.geom |asset.name   |asset.stories|asset.constr…|hazard.geom |hazard.kaiju…|hazard.seve…|agg.geom|agg.name    |
+---------+-------------+-----------+-------------+-------------+-------------+------------+-------------+------------+--------+------------+
|Polygon… |0            |Polygon…   |Central Perk |1            |brick        |            |             |            |Polygon…|Manhattan   |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |0            |Polygon…   |The Hayden   |5            |stone        |            |             |            |Polygon…|Manhattan   |
|         |             |           |Planetarium  |             |             |            |             |            |        |            |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |0            |Polygon…   |That Cafe    |1            |stone        |            |             |            |Polygon…|Manhattan   |
|         |             |           |From Seinfeld|             |             |            |             |            |        |            |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |0            |Polygon…   |The Met      |15           |steel        |            |             |            |Polygon…|Manhattan   |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |42           |Polygon…   |Statue of    |22           |stone        |LineString… |Gauwdzulla   |7.0         |Polygon…|Liberty     |
|         |             |           |Liberty      |             |             |            |             |            |        |Island      |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |0            |Polygon…   |Natural      |6            |stone        |            |             |            |Polygon…|Manhattan   |
|         |             |           |History      |             |             |            |             |            |        |            |
|         |             |           |Museum       |             |             |            |             |            |        |            |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |0            |Polygon…   |Ed Sullivan  |15           |brick        |            |             |            |Polygon…|Manhattan   |
|         |             |           |Theatre      |             |             |            |             |            |        |            |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |24           |Polygon…   |NYSE         |8            |stone        |LineString… |Gauwdzulla   |4.0         |Polygon…|Manhattan   |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |24           |Polygon…   |Tiffany and  |8            |brick        |LineString… |Gauwdzulla   |4.0         |Polygon…|Manhattan   |
|         |             |           |Co           |             |             |            |             |            |        |            |
|         |             |           |             |             |             |            |             |            |        |            |
|Polygon… |6            |Polygon…   |Federal      |16           |concrete     |LineString… |Gauwdzulla   |2.0         |Polygon…|Manhattan   |
|         |             |           |Reserve      |             |             |            |             |            |        |            |
+---------+-------------+-----------+-------------+-------------+-------------+------------+-------------+------------+--------+------------+
```

Let's say the borough names are so useful, we always want to include them in our results.
In your `project.ini` file, find the `[model kaiju_stomp]` section and add the following lines:

```none
aggregation-layer = nyc_boroughs
include-aggregation-layer-in-output = true
```

Now try running the following (slightly simplified) command and check that we still get the
borough included in our output:
`riskscape model run kaiju_stomp -f csv`

