.. _models-intro:

# How RiskScape models work

## Before we start

This tutorial is aimed at new users who want an introduction to how risk models work in RiskScape.
We expect that you:
- Have been through the :ref:`getting-started` guide and have a basic idea of what RiskScape is about.
- Are comfortable running shell commands from a command prompt.
  If you are inexperienced at using the command-line, try reading :ref:`cli-help` first.
- Have a basic understanding of geospatial data and risk analysis.
- Have :ref:`gis-application` (e.g. QGIS) installed that you can use to view shapefile and GeoTIFF files.

.. note::
    ArcGIS users should read :ref:`arcgis_workarounds` before proceeding.
    It lists tips to ensure that geospatial data is projected correctly in ArcGIS.

The aim of this tutorial is to introduce the underlying concepts behind running models in RiskScape,
and familiarize new users with the RiskScape Command-Line Interface (CLI).

For simplicity, this tutorial will _not_ delve into much detail around creating bookmarks, writing your own functions,
or building your own models or pipelines from scratch.
These topics will all be covered in later tutorials.

.. warning::
  Ensure that you have the latest copy of RiskScape installed on your system. For directions on how to install RiskScape, see :ref:`installation`

## Getting started

### Setup

Click [here](../models-intro.zip) to download the input data we will use in these examples.
Unzip the file into the :ref:`top_level_dir` where you keep your RiskScape projects.
For example, if your top-level projects directory is `C:\Users\%USERNAME%\RiskScape\Projects\`,
then your unzipped directory will be:  
`C:\Users\%USERNAME%\RiskScape\Projects\models-intro`

Open a command prompt and use `cd` to change directory to where you unzipped the files, e.g.

```none
cd models-intro
```

You will use this command prompt to run the RiskScape commands in this tutorial.

The input data files this tutorial will use are all in the `models-intro\data` sub-directory.

.. note::
    This input data was provided by `Earth Sciences New Zealand <https://earthsciences.nz/>`_ and is based on published research papers.
    The regional map of Samoa was sourced from the `PCRAFI <http://pcrafi.spc.int/>`_
    (Pacific Risk Information System) website.
    The data files have been adapted slightly for this tutorial.

### Input data

In this tutorial you will run several RiskScape models that determine tsunami exposure in Samoa.
We will use similar source data to the :ref:`getting started guide <Upolu_data>`,
modelled on the 2009 South Pacific Tsunami (SPT) event.

A RiskScape :ref:`model <models>` involves combining geospatial layers together, and then performing risk analysis.
The name of each layer reflects how RiskScape will use the layer in the model.
The RiskScape models in this tutorial involve the following layers:

- _Exposure-layer_: This contains the elements-at-risk, which is our building dataset, `Buildings_SE_Upolu.shp`.
  This shapefile was based on [OpenStreetMap](https://www.openstreetmap.org) data for south-eastern Upolu.
- _Hazard-layer_: This contains the hazard being modelled, which is a tsunami inundation map based on the 2009 SPT.
  We are using two different hazard layers, `MaxEnv_All_Scenarios_10m.tif` and `MaxEnv_All_Scenarios_50m.tif`,
  which use a 10-metre and 50-metre grid-size respectively.
  You can read more about how these hazard layers were produced [here](https://www.mdpi.com/2624-795X/2/2/4/htm).
- _Area-layer_: This contains regional boundaries, which can be optionally used to collate the model results,
  such as reporting the number of exposed buildings in each region.
  We will use the boundaries of the Samoa electoral constituencies (`Samoa_constituencies.shp`) in this example, which was sourced
  from the [PCRAFI](http://pcrafi.spc.int/) website.

.. note::
    RiskScape models also support an optional *Resource-layer*, which contains any other supplementary information required for risk analysis.
    For example, a resource-layer containing the soil type or water table depth might be used in a liquefaction model.
    This tutorial will not cover resource-layers at all.

### RiskScape Projects

Alongside the `data` sub-directory, you should notice there is also a `project.ini` file in your unzipped directory.
This file is called the RiskScape _project_ file, and it contains all the instructions that RiskScape needs to run your models.

:ref:`projects` are a way to organize your RiskScape models, a bit like a work-space.
You can use separate projects when you want to model different things.

The project file is a plain-text file in the [INI format](https://en.wikipedia.org/wiki/INI_file).
INI files simply contain sets of `key = value` pairs that store configuration settings and values.

Open the `project.ini` file in a text editor, such as Notepad or `gedit`.

We have pre-populated this project file with all the models we will use in this tutorial.
In later tutorials we will explain how to build your own models and project files from scratch.

.. tip::
    RiskScape defaults to using the ``project.ini`` file in the current working directory that you run the ``riskscape`` command from.
    If you want RiskScape to load a different project file, you can specify the file to use on the command-line.
    Refer to :ref:`specify_project` for more details.

## Running a model

RiskScape models provide a flexible framework that let you customize aspects of the risk analysis workflow,
such as using different input data, changing the analysis that is performed, or varying other
assumptions that your model might make.

RiskScape provides a command-line wizard that can help you to build a risk analysis model by answering a few questions.
You can save the wizard answers in an INI file, which lets you re-run your model later.
We will look more at building models in the wizard in the next tutorial.

The customizable aspects of a model are called the model's _parameters_.
Changing the model parameters lets you change the behaviour of the model on the fly.

Conceptually, what we will be doing in this tutorial will look like this:

.. mermaid::

    graph LR
      WIZ("Wizard answers");
      MODEL("Risk Model");
      WIZ -- Saved config --> MODEL;
      PARAMS("Parameters<br/>to vary") -.-> MODEL;
      MODEL --> RUN("Model run<br/>command");
      RUN --> EIT("Event Impact<br/>results");

      %% Styling
      class PARAMS rs-box-optional
      class WIZ,MODEL,PARAMS, rs-box
      class RUN rs-box-navy
      class EIT rs-box-green
      style RUN color:#FFF

We will take saved wizard answers, optionally modify the model's parameters,
and then use a `riskscape` command to 'run' the risk model.
Running the risk model will produce results file(s), which tell us the impact of the hazard event.

### A simple RiskScape model

In the `project.ini` file, there is a `basic-exposure` model that looks something like this:

```ini
[model basic-exposure]
description = Simple example of a RiskScape model
framework = wizard
input-exposures.layer = data/Buildings_SE_Upolu.shp
input-hazards.layer = data/MaxEnv_All_Scenarios_50m.tif
sample.hazards-by = CLOSEST
analysis.function = is_exposed
```

This model was produced by using the RiskScape wizard, so it contains the saved answers to wizard questions.
Each line is an INI file key-value, pair separated by an `=`.
The left-hand-side (i.e. the *key*) is the name of the model *parameter*, and the right-hand-side
is the *value* that will be used in the risk analysis model.

.. note::
    This ``"PARAMETER_NAME = VALUE"`` format is important because it determines how your risk model will behave.
    Changing the value of the model's parameters will change the results that the model produces,
    which we will look at soon.

The model parameters correspond to decisions that RiskScape needs to make at various points during the model processing.
A high-level overview of the processing behind our RiskScape model looks like this:

.. mermaid::

    graph LR

      EL("Exposure-layer<br/>(building data)");
      HL("Hazard-layer<br/>(tsunami inundation)");

      GS("Geospatial matching<br/>(sampling)");
      EL --> GS;
      HL --> GS;
      PY("Python function");
      GS --> CA("Consequence Analysis");
      PY -.-> CA;
      CA --> REP("Report Event<br/>Impact results");

      %% Styling
      class EL,HL rs-box-navy
      style EL color:#FFF
      style HL color:#FFF
      class CA,GS rs-box
      class REP rs-box-green
      class PY rs-box-optional

So for example, the `input-exposures.layer` parameter specifies the input data that we will use for our exposure-layer (our buildings shapefile).
Similarly, the `input-hazards.layer` parameter specifies the hazard-layer input data for our model (our tsunami inundation GeoTIFF).

The `sample.hazards-by` parameter specifies how to spatially *sample* the hazard-layer.
This determines how RiskScape geospatially matches elements-at-risk in your exposure-layer to the hazard-layer.

We will look at spatial sampling in more detail in subsequent tutorials.
For now, 'closest' just means that RiskScape will match any hazard value that intersects an element-at-risk.

Finally, the `analysis.function` parameter specifies a function that will determine the impact, or consequence, that
the hazard has on each element-at-risk.

In general, you will want to use your own Python function to do the analysis.
For now, we are using a built-in RiskScape function called `is_exposed`, which simply determines whether or not a building is exposed to the hazard.
The resulting consequence will be `1` (representing true) if the building is exposed, and `0` (representing false) if not.

This `basic-exposure` model does not change how the event impact results are reported, but we will look at that more later.

### Running the model

To run this `basic-exposure` model, enter the following command into your terminal prompt.

```none
riskscape model run basic-exposure
```

The model should take a few seconds to run.
RiskScape prints out a progress meter as it runs your model.
The numbers displayed correspond to rows of data processed.
For more detail on what these numbers mean, refer to :ref:`progress_monitor`.

.. tip::
    If the RiskScape command produced an error, try checking that ``riskscape -V`` runs without
    any errors, and that the current directory in your command prompt is in the unzipped ``models-intro``
    directory that contains the ``project.ini`` file.

When RiskScape has finished it will display the output file produced, which contains the results of the model.
By default, RiskScape saves its results to an `output/MODEL_NAME/TIMESTAMP/` sub-directory
within the current working directory, e.g. `./output/basic-exposure/2021-11-10T15_09_25/event_impact_table.shp`

Open the `event_impact_table.shp` in your preferred GIS application, e.g. QGIS.
At first glance, this data looks exactly the same as the exposure-layer we started with.
However, if you open the attribute table, you will see each element-at-risk now has a
`hazard` value, which is the tsunami inundation height, and a `consequence`, which is `0` or `1` depending
on whether the building was exposed to the tsunami.

### Changing the model outputs

RiskScape commands can also accept additional CLI _options_ that change the way the command behaves.

For the `riskscape model run` command, CLI options can change the way results are generated.
The CLI options always have dashes (i.e. `-` or `--`) in front of them,
and are always specified on the _end_ of the `riskscape model run` command.

For example, if you want to change the default output location that RiskScape uses,
you can use the `--output`, or `-o` command-line option.
Try running the following command, which will save the output files directly to a `results_dir` sub-directory.

```none
riskscape model run basic-exposure --output results_dir
```

By default, RiskScape will never overwrite any existing results files when you use the `--output` option.
Try running the same command again - this time, the filename gets a number appended, i.e. `analysis-1.shp`.

You can use the `--replace`, or `-r` option to always overwrite the previous results from a model. For example:

```none
riskscape model run basic-exposure --output results_dir --replace
```

Because the results for the `basic-exposure` model contain geometry,
RiskScape will default to saving the results in shapefile format.

To change the default file format that results get saved in, use the `--format`, or `-f` option.
For example, to save the results as Comma-Separated Values (CSV) file, use:

```none
riskscape model run basic-exposure --format csv
```

You can use `--help` on the end of a RiskScape command to find out more about what CLI options are supported.
See if you can work out what other formats RiskScape can save your results in.

```none
riskscape model run --help
```

.. tip::
    Many RiskScape CLI options have both a *long* option (e.g. ``--format``) and a *short* alternative (e.g. ``-f``).
    The short option is always a single character and always has a *single* dash.
    Whereas the long option will always be full words and always has *two* dashes.

### Reporting the model results

The `basic-exposure` model produces an _Event Impact Table_,
which simply contains the consequence, if any, that the hazard had on each element-at-risk.

We can customize how the event impact table is saved, which lets us report the
effects of the hazard in a range of useful ways.

Now let's look at the `exposure-reporting` model in the `project.ini` file.
This is exactly the same as the `basic-exposure` model we ran previously,
except it has extra parameters to *report* the event impact table in different ways.

```ini
[model exposure-reporting]
description = Model with additional reporting options that alter how the results are saved
framework = wizard
input-exposures.layer = data/Buildings_SE_Upolu.shp
input-hazards.layer = data/MaxEnv_All_Scenarios_50m.tif
sample.hazards-by = CLOSEST
analysis.function = is_exposed
report-event-impact.filter = consequence = 1
reports.name[0] = summary
report-summary.group-by[0] = 'Total' as Results
report-summary.aggregate[0] = count(hazard) as Number_Exposed
report-summary.aggregate[1] = count(exposure) as Total_buildings
report-summary.aggregate[2] = max(hazard) as Max_hazard
report-summary.aggregate[3] = mean(hazard) as Average_hazard
report-summary.format = csv
```

Try running the `exposure-reporting` model using the following command:

```none
riskscape model run exposure-reporting
```

This model produces two results files in a `output/exposure-reporting/TIMESTAMP/` sub-directory:
- `event-impact.shp`: This contains _only_ the elements-at-risk that were exposed to the hazard.
  Try opening the file in QGIS. It makes it a bit easier to visualize which buildings in the original exposure-layer were affected by the tsunami.
- `summary.csv`: This summarizes the effects of the tsunami, reporting the total number of buildings that were exposed.
  The maximum and average tsunami inundation depths for affected buildings has also been included.

Try opening the `summary.csv` file, either in a spreadsheet application or
use the `more "FILENAME"` command to display the file contents from the command prompt.
It should look like this:

```none
more "output/exposure-reporting/2021-11-11T11_35_02/summary.csv"
Results,Number_Exposed,Total_buildings,Max_hazard,Average_hazard
Total,2059,6260,8.218506813049316,2.5090660887936354
```

.. tip::
    Forward slashes in file-paths generally work OK in the Windows Command Prompt, as long as
    you surround them in double-quotes, e.g. ``"output/some-file.csv"``.
    This means you can copy-paste the results filename from the URI that RiskScape displays.
    Simply select the text and use ``Ctrl`` + ``c`` and ``Ctrl`` + ``v`` to copy-paste in the Windows
    Command Prompt.

## Model parameters

So far we have been customizing the model's behaviour using parameters saved in the INI file.
However, you can also change the model's parameters on the fly, using the command line.

To change a parameter in a model we use the `--parameter` CLI option, or `-p` for short.

.. tip::
    You can view all the parameters that are available in a model using the ``riskscape model list`` command.

### Changing the input data

As we saw in the :ref:`getting-started` guide, we can change the hazard-layer that our model uses
from a 50-metre grid to a 10-metre grid.

When changing a parameter, you specify the parameter's value in the format ``-p "PARAMETER_NAME=VALUE"``.
In this case, the hazard-layer input parameter is called `input-hazards.layer`,
and the value is the 10m grid file we want to use, `data/MaxEnv_All_Scenarios_10m.tif`.

Try entering the following command to use the 10m grid hazard-layer in our model instead.

```none
riskscape model run exposure-reporting -p "input-hazards.layer=data/MaxEnv_All_Scenarios_10m.tif"
```

Now open the `summary.csv` file that was produced. It should look like this:

```none
Results,Number_Exposed,Total_buildings,Max_hazard,Average_hazard
Total,1568,6260,5.786825180053711,1.673170207030311
```

The same number of buildings were run through the model (i.e. `Total_buildings`), but now fewer buildings
are reported as being exposed to the tsunami (i.e. `Number_exposed`) compared with previously.

### Input data attributes

You can think of the data in a RiskScape model as being a bit like a big spreadsheet -
each row of data represents an element-at-risk from the exposure-layer, and the columns represent _attributes_ associated with the element-at-risk.
As the input data moves through the model, the rows and columns can change.

The model data starts off with the exposure-layer input data.
So any building attribute that is present in the input data can be used later in the model.

You can use the following command to display more information about our buildings database file, `data/Buildings_SE_Upolu.shp`:

```none
riskscape bookmark info "data/Buildings_SE_Upolu.shp"
```

The attributes that are present in the file will be displayed at the top of the output, e.g.

```
Location : file:///C:/RiskScape_Projects/models-intro/data/Buildings_SE_Upolu.shp
  Attributes  :
    the_geom[MultiPolygon[crs=EPSG:32702]]
    Use_Cat[Text]
    Cons_Frame[Text]
    area[Floating]
    perimeter[Floating]
  Axis-order  : long,lat / X,Y / Easting,Northing
  CRS code    : EPSG:32702
  CRS (full)  : PROJCS["WGS_1984_UTM_Zone_2S",
  GEOGCS["GCS_WGS_1984",
    DATUM["D_WGS_1984",
      SPHEROID["WGS_1984", 6378137.0, 298.257223563]],
    PRIMEM["Greenwich", 0.0],
    UNIT["degree", 0.017453292519943295],
    AXIS["Longitude", EAST],
    AXIS["Latitude", NORTH]],
  PROJECTION["Transverse_Mercator"],
  PARAMETER["central_meridian", -171.0],
  PARAMETER["latitude_of_origin", 0.0],
  PARAMETER["scale_factor", 0.9996],
  PARAMETER["false_easting", 500000.0],
  PARAMETER["false_northing", 10000000.0],
  UNIT["m", 1.0],
  AXIS["x", EAST],
  AXIS["y", NORTH]]
  Summarizing...
  Row count   : 6260
  Bounds      : EPSG:32702 [410019.3061 : 454991.3475 East, 8445166.5924 : 8464016.1628 North] (original)
  Bounds      : EPSG:4326 [-171.8334 : -171.4166 East, -14.0639 : -13.8924 North]
```

We can see the building database has the attributes `the_geom`, `Use_Cat`, `Cons_Frame`, `area`, and `perimeter`.

RiskScape groups together attributes that come from the same input layer.
You can refer to an attribute from a given input layer in your RiskScape model using the format `LAYER_NAME.ATTRIBUTE_NAME`.
For example, to refer to the `Use_Cat` *exposure*-layer attribute you would use `exposure.Use_Cat`.

### Changing the output data

By changing model parameters, we can also change how the model results are reported.

For example, currently our model is aggregating (i.e. collating or summarizing) the results across _all_ buildings exposed.
Instead, we can group the results by the building attribute we are interested in,
such as its use category (i.e. `exposure.Use_Cat`)  or its construction type (i.e. `exposure.Cons_Frame`).

The model parameter that groups the results for the `summary.csv` report is `report-summary.group-by[0]`.
The `[0]` part of the parameter just means you can group the results by multiple things.

Try running the following command, which groups against the building's use-category attribute (`exposure.Use_Cat`).

```none
riskscape model run exposure-reporting -p "report-summary.group-by[0]=exposure.Use_Cat"
```

Now look at the `summary.csv` file that was produced. It should contain the following:

```none
Use_Cat,Number_Exposed,Total_buildings,Max_hazard,Average_hazard
Outbuilding,623,1343,7.879426002502441,2.331427180173883
Residential,930,1862,7.718549728393555,2.2250018423603426
Industrial,6,8,3.821390390396118,2.7654923796653748
Religious - not verified,0,2,,
Fale,111,146,8.096136093139648,2.7053862896051495
Commercial,14,51,7.13636589050293,3.719742706843785
Commercial - not verified,1,3,1.5244611501693726,1.5244611501693726
Religious,28,60,5.8724517822265625,1.8323415092059545
Hotel - not verified,0,1,,
Community - not verified,2,6,5.375617504119873,5.0066187381744385
Hotel,198,226,8.218506813049316,3.390313490472659
Education,12,12,5.941905975341797,3.395870625972748
Tourist Fale,107,157,6.352619171142578,3.8892281768477965
Commericial - not verified,1,1,5.397562026977539,5.397562026977539
Community,23,26,6.209683418273926,3.2658567169438237
,3,2356,1.9251530170440674,1.3111796379089355
```

We now get a breakdown of the buildings exposed to the tsunami based on what the building is used for.

### Changing other model assumptions

The model's `summary.csv` report currently includes buildings that were exposed to _any_ tsunami inundation.
We could filter the results so that only buildings that were exposed to a certain hazard threshold are counted in the summary.

The model parameter that filters the results for the `summary.csv` report is `report-summary.filter`.
The inundation depth appears in the model data as the `hazard` attribute, so `hazard >= 0.5`
would only include results that have a minimum inundation depth of 0.5m.

Try running the following command, to filter the results based on the inundation depth.


```none
riskscape model run exposure-reporting -p "report-summary.filter=hazard >= 0.5"
```

Now look at the `summary.csv` file that was produced.

```none
Results,Number_Exposed,Total_buildings,Max_hazard,Average_hazard
Total,1714,1714,8.218506813049316,2.977457029677606
```

You can see that the `Number_Exposed` is now less than it was previously (2059).
It has now excluded 345 buildings that were exposed to an inundation depth of 0.5m or less.

.. note::
    The ``Total_buildings`` used to be ``6260``, but now it is ``1714``, the same as ``Number_Exposed``.
    This is because the filter condition applies *before* the ``Total_buildings`` count is calculated.
    There are ``4546`` buildings that do not meet the filter condition, i.e. that are unexposed or exposed to *less* than 0.5m of inundation.
    These unexposed buildings are now excluded from the summary results, and so no longer included in the ``Total_buildings`` count.

## Aggregating by area-layer

A useful way to view your model results is to aggregate them by a regional boundary of interest, i.e. an _area-layer_.
For example, you might be interested in how the impact from a hazard affects a specific district or particular towns.

We will look at how our tsunami hazard affects the different regions of south-eastern Upolu
by using an area-layer of the Samoan electoral constituencies (`Samoa_constituencies.shp`).

Try running the `exposure-by-region` model using the following command.

```none
riskscape model run exposure-by-region
```

This model produces an `event-impact.csv` results file in the `output/exposure-by-region/TIMESTAMP/` sub-directory.
This `event-impact.csv` contains our event impact table aggregated by region, which looks like this:

```
Region,Exposed_Status,Total_buildings
Aleipata Itupa i Lalo,0,332
Aleipata Itupa i Lalo,1,531
Aleipata Itupa i Luga,0,210
Aleipata Itupa i Luga,1,345
Anoamaa East,0,322
Anoamaa West,0,32
Falealili,0,1013
Falealili,1,749
Lepa,0,232
Lepa,1,288
Lotofaga,0,277
Lotofaga,1,146
Safata,0,595
Siumu,0,682
Vaa o Fonoti,0,405
Vaimauga West,0,101
```

This time there are up to two rows for each region - one row for exposed buildings (i.e. `Exposed_Status == 1`)
and another row for unexposed buildings (i.e. `Exposed_Status == 0`)

.. note::
    Many of the electoral constituencies from the area-layer are not included in the results - only
    regions that overlapped with our exposure-layer's elements-at-risk were included, and our
    exposure-layer only covers south-eastern Upolu.

## Specifying an analysis function

So far we have been using a built-in RiskScape function called `is_exposed` to analyse the consequence that the hazard-layer has on each element-at-risk
(which is simply `1` if the building is exposed, `0` if not).

You can define your own Python function that RiskScape will use to analyse the impact of the hazard.
This means your RiskScape model can do whatever risk analysis you want it to.

In your project directory there is a `functions/exposure.py` file that contains the following Python code:

```python
def function(building, hazard_depth):
    if hazard_depth is None or hazard_depth <= 0:
        return 'Not exposed'

    if hazard_depth > 3.0:
        status = 'Exposure >3.0m'
    elif hazard_depth > 2.0:
        status = 'Exposure >2.0m to <=3.0m'
    elif hazard_depth > 1.0:
        status = 'Exposure >1.0m to <=2.0m'
    else:
        status = 'Exposure >0.0m to <=1.0m'

    return status
```

This is a simple Python function that assigns a status to each element-at-risk based on the depth of inundation that the building was exposed to.

Note that in order to use a Python function in your model, we have to first tell RiskScape about the function.
In the `project.ini` file, we have the following entry that adds the Python function to RiskScape:

```ini
[function exposure_status]
description = A simple function to evaluate a building's exposure status to tsunami inundation
location = functions/exposure.py
argument-types = [ building: anything, hazard_depth: nullable(floating) ]
return-type = text
```

This tells RiskScape things like the function's name (`exposure_status`) and the Python file to use.
You will learn more about what this INI definition means, and how to add your own functions, in a later tutorial.

Now try using this `exposure_status` function in a model by running the following command:

```none
riskscape model run exposure-by-region -p "analysis.function=exposure_status"
```

Open the `event-impact.csv` file produced.
You can now see a more detailed breakdown of how many buildings in each region were exposed
to different tsunami inundation depths.

```
Region,Exposed_Status,Total_buildings
Aleipata Itupa i Lalo,Exposure >0.0m to <=1.0m,89
Aleipata Itupa i Lalo,Exposure >1.0m to <=2.0m,154
Aleipata Itupa i Lalo,Exposure >2.0m to <=3.0m,166
Aleipata Itupa i Lalo,Exposure >3.0m,122
Aleipata Itupa i Lalo,Not exposed,332
Aleipata Itupa i Luga,Exposure >0.0m to <=1.0m,74
Aleipata Itupa i Luga,Exposure >1.0m to <=2.0m,21
Aleipata Itupa i Luga,Exposure >2.0m to <=3.0m,42
Aleipata Itupa i Luga,Exposure >3.0m,208
Aleipata Itupa i Luga,Not exposed,210
Anoamaa East,Not exposed,322
Anoamaa West,Not exposed,32
Falealili,Exposure >0.0m to <=1.0m,225
Falealili,Exposure >1.0m to <=2.0m,166
Falealili,Exposure >2.0m to <=3.0m,150
Falealili,Exposure >3.0m,208
Falealili,Not exposed,1013
Lepa,Exposure >0.0m to <=1.0m,41
Lepa,Exposure >1.0m to <=2.0m,31
Lepa,Exposure >2.0m to <=3.0m,76
Lepa,Exposure >3.0m,140
Lepa,Not exposed,232
Lotofaga,Exposure >0.0m to <=1.0m,44
Lotofaga,Exposure >1.0m to <=2.0m,22
Lotofaga,Exposure >2.0m to <=3.0m,38
Lotofaga,Exposure >3.0m,42
Lotofaga,Not exposed,277
Safata,Not exposed,595
Siumu,Not exposed,682
Vaa o Fonoti,Not exposed,405
Vaimauga West,Not exposed,101
```

## Recap

Let's review some of the key points we have covered so far:

- All the instructions RiskScape needs to run a model are stored in a plain-text `project.ini` INI file.
- A RiskScape model combines together various geospatial layers and performs some risk analysis.
- RiskScape models are run from the command line. You can run a saved model multiple times, and change the file format or location where the model results are saved.
- You can vary a model's parameters in order to change the input data files used, or other aspects of how the model works.
- A RiskScape model produces an event impact table. You can customize how the event-impact table is reported, or save the same model results in multiple different ways.
- Aggregation is a useful way to report the model results in a meaningful way. The model's area-layer lets you aggregate results by a regional boundary.
- The RiskScape model uses a function to analyse the consequence the hazard has on the exposure-layer. You can specify your own Python function to perform this consequence analysis.

.. tip::
    The models used in this tutorial are generally applicable for reuse.
    By changing the exposure-layer and hazard-layer, you can use them to model how any elements-at-risk
    are exposed to a hazard, particularly a hazard-layer in ``.asc`` or ``.tif`` format.

Once you feel comfortable with running models, you could try the next tutorial on :ref:`wizard-how-to`.

## Extra for experts

If you want to play around with running models a little more, you could try the following:

- Try running the `basic-exposure` model so that the results are produced in GeoJSON format.
- Try grouping the `exposure-reporting` results by the `Cons_Frame` exposure-layer attribute (i.e. `exposure.Cons_Frame`).
  Open the `summary.csv` file that gets produced and see what difference this makes to the results.
- Try running the `exposure-reporting` with a filter parameter so that the `event-impact.shp` results only
  includes buildings that were exposed to 1m of inundation or greater.
  Hint: the filter will apply to the _event-impact_ report (i.e. `report-event-impact.filter`) rather than
  the _summary_ report (i.e. `report-summary.filter`).
  Open the `event-impact.shp` results in QGIS and compare it with the original building database layer.
- Use the `riskscape bookmark info` to work out what attributes are present in the `data/Samoa_constituencies.shp` file.
  Open the shapefile's attribute table in QGIS and check that the attributes match RiskScape's output.
- Try running the `exposure-by-region` model with `-p "report-event-impact.group-by[2]=exposure.Cons_Frame"`.
  This groups the results by the building construction type, as well as the region and consequence.
  Look at the `event-impact.csv` results file that gets produced.
  What are most affected buildings constructed from? Does it vary at all by region?
- Try modifying the `project.ini` file so that the `exposure-by-region` always uses the `exposure_status` function by default,
  rather than the `is_exposed` function.
  Run the `exposure-by-region` model again (without specifying any model parameters on the command-line),
  and check the `event-impact.csv` results file now contains the `Exposure >X.0m` consequences.
- Open the `functions/exposure.py` file in a text editor and try modifying it.
  Add an extra `elif` statements so that the `>0.0m to <=1.0m` status is split into two: `>0.5m to <=1.0m` and `>0.0m to <=0.5m`.
  Run the `exposure-by-region` model again and check that you see the new status in the results file.
- Enter `riskscape model list` and look at the parameters for the `exposure-by-region` model.
  Currently the model sorts the output by region name. See if you can work out the model parameter that sorts the results.
  Try running the model so that it sorts the output by the `Total_buildings` value instead.
  Open the `event-impact.csv` file that gets produced and check what order the results are in.
- Try running the `exposure-by-region` model so it reports the exposure status for _all_ of Upolu, rather than by region.
  This is a little awkward to do currently with wizard models, but try changing the `report-event-impact.group-by[0]` parameter to group by constant value, e.g. `'Upolu' as Region`.
  See what this does to the `event-impact.csv` results that get produced.

