.. _table_pipelines:

# How to use table models

A RiskScape model normally takes exposure and hazard input data from *separate* files and geospatially combines them together.
However, sometimes your exposure and hazard input data may already be combined together in a *single* file.

In this case, we want to skip all the geospatial matching and just apply a given function to our exposure and hazard values. 
We call this a *table* model, because it is essentially using a single 'table' of input data.

.. tip::
    This page is all about the special case where you have a *single* input file that contains *both* your exposure and hazard data.
    If this does not apply to you, then you can safely skip this page.

## Before we start

This tutorial is aimed at RiskScape users who have already combined their geospatial data and want to run it through a model.
We expect that you:
- Have been through the introductory tutorials on using RiskScape (i.e. :ref:`getting-started`, etc)
  and are comfortable with running models and changing their parameters.
- Have a basic understanding of what a :ref:`pipeline model <pipelines>` is, and can write simple RiskScape :ref:`expressions`.
- Are comfortable working with geospatial data and risk analysis models.

The aim of this tutorial is to build a pipeline for a table model from scratch.

## Setup

Click [Here](../table-model.zip) to download the input data we will use in these examples.
Unzip the file into a new directory, e.g. `C:\RiskScape_Projects\`.

Open a command prompt and `cd` to the directory where you unzipped the files, e.g.

```none
cd C:\RiskScape_Projects\table-model
```

You will use this command prompt to run the RiskScape commands in this tutorial.

The input data files this tutorial will use are all in the `table-model\data` sub-directory.
This is similar to the :ref:`Upolu tsunami data <Upolu_data>` that we used in the getting started tutorials,
except that our building database has been already matched to the hazard-layer data.

.. note::
    This input data was provided by `Earth Sciences New Zealand <https://earthsciences.nz/>`_, as well as the `PCRAFI <http://pcrafi.spc.int/>`_
    (Pacific Risk Information System) website.
    The data files have been adapted slightly for this tutorial.

## Running a simple table model

The `project.ini` file contains the following `basic-table` model:

```ini
[model basic-table]
framework = pipeline
location = basic-pipeline.txt
description = A simple model to apply a function to a "table" of data
param.relation = 'Buildings_SE_Upolu_with_inundation'
param.function = Samoa_Building_Fragility_DS5(exposure, hazard)
param.exposure_attributes = *
param.hazard_attributes = inundation
```

This defines a model using a *pipeline*, as well as defining *parameters* that go into the model.

.. tip::
    You don't *have* to understand all the specifics of the pipeline yet.
    Just know that you can run a pipeline model and change its parameters just like any other model.
    If you want to understand more about how a pipeline model works, try going through
    the :ref:`pipelines_tutorial` and :ref:`parameterized_pipelines` tutorials.


This `basic-table` model takes an input data file (called a *relation* here)
and a function, and simply applies the function to the data to produce a *consequence*.
In this case, the data is `data/Buildings_SE_Upolu_inundation.shp` and the function is `Samoa_Building_Fragility_DS5`,
which calculates the probability of a building being in damage state 5 (i.e. complete structural collapse)

Try running this model now, using the command:

```none
riskscape model run basic-table
```

This should produce a `event_loss_table.shp` results file.
Open this file in QGIS and take a look. You should see that each building now has a
`consequence` value calculated for it.

## Changing table model parameters

You can change the parameters that the model uses, just like any model.
We need to tell this model which attributes in the input data represent our exposure data (i.e. the element-at-risk),
and which ones represent the hazard.

Our function (`Samoa_Building_Fragility_DS5`) only actually needs the construction framing attribute (i.e. `Cons_Frame`).
Try running the following command to run the model with just this single exposure-layer attribute:

```none
riskscape model run basic-table -p "exposure_attributes=Cons_Frame" 
```

Notice that the output file has changed from a shapefile to a CSV file.
This is because we are no longer including the input geometry as part of our exposure-layer data.

.. note::
  In general, input data for table models may or may not contain geometry representing the exposure-layer.
  The distinguishing feature of a table model is that the exposure and hazard information is in a *single* file.

Try including the input geometry as well by running the following command:

```none
riskscape model run basic-table -p "exposure_attributes={ the_geom, Cons_Frame }"
```
.. tip::
    When you specify multiple attributes you have to surround them with ``{`` ``}`` braces.
    This is because it needs to be a valid RiskScape :ref:`expression <expressions>`.

## Table model aggregation

You can also use RiskScape to produce an *aggregated* summary of your results.
The `project.ini` file also contains the following `table-aggregation` model:

```ini
[model table-aggregation]
framework = pipeline
location = aggregation-pipeline.txt
description = A simple model to apply a function to a "table" of data
param.relation = 'data/Buildings_SE_Upolu_inundation.shp'
param.function = Samoa_Building_Fragility_DS5(exposure, hazard)
param.exposure_attributes = *
param.hazard_attributes = inundation
param.filter = true
param.aggregation_layer = 'data/Samoa_constituencies.shp'
param.buffer_distance = 10000
param.aggregate_by = region
param.aggregate_function = mean(consequence)
```

This uses the regional boundaries in `data\Samoa_constituencies.shp` to aggregate the model's results.
Try running the model using:

```none
riskscape model run table-aggregation
```

Open the resulting `event_impact.shp` file in QGIS.
This contains the model results broken down on a regional basis.
The average consequence (probability of being in damage state 5) is reported for each region.

### Changing the aggregation

We can change the model parameters to present the results in different ways.
For example, instead of summarizing by region we could summarize by construction type.
Try running the following command:

```none
riskscape model run table-aggregation -p "aggregate_by=exposure.Cons_Frame" 
```

This time it produces a CSV results file instead of a shapefile.
This is because we are now ignoring the regional geometry.

We could aggregate by both region name *and* construction type, by using the following command:

```none
riskscape model run table-aggregation -p "aggregate_by={ region.Region, exposure.Cons_Frame }"
```

.. note::
    Because we are aggregating by *multiple* attributes, we need to enclose the
    attributes in ``{ }`` braces again, so that it is a valid RiskScape expression.

If we wanted a single overall total across all regions, we could get that by using the following command:

```none
riskscape model run table-aggregation -p "aggregate_by='Total'"
```

### Changing the aggregated result

So far we have changed what we are aggregating *by*.
We can also change what aggregated result we calculate using the `aggregate_function` parameter.

The `aggregate_function` parameter takes an expression similar to a spreadsheet formula, e.g. `mean(consequence)` reports the average consequence.
We could change `mean` to another function, such as `min`, `max`, `sum`, `count`, or `stddev`.
For example to work out the standard deviation (which also includes the mean), use the command:

```none
riskscape model run table-aggregation -p "aggregate_function=stddev(consequence)"
```

There are also other aggregation functions, but some are slightly trickier to use.
For example, to calculate the ninety-fifth percentile result, use the command:

```none
riskscape model run table-aggregation -p "aggregate_function=percentile(consequence, 95)"
```

Just like the other parameters, you can specify multiple values if you use `{` `}` braces.
For example, to calculate both the mean *and* a percentile, use the command:

```none
riskscape model run table-aggregation -p "aggregate_function={ mean(consequence), percentile(consequence, 95) }"
```

## Recap

This tutorial has shown you how you can apply a RiskScape model to a *single* input source,
i.e. a single table of data. You can reuse these example models for your own projects - simply
change the `relation` and `function` parameters that the models use.

