/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2008-2014, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *   DbfHeader logic from ShapefileWriter added to static method for reuse from RiskScape and license compliance
 */
package org.geotools.data;

import java.math.BigInteger;
import java.util.Calendar;

import org.geotools.data.shapefile.dbf.DbaseFileException;
import org.geotools.data.shapefile.dbf.DbaseFileHeader;
import org.geotools.data.shapefile.shp.ShapefileWriter;

/**
 * Logic for writing dbf header based on java types, lifted from {@link ShapefileWriter} for reuse by RiskScape
 */
public class ShapefileWriting {

  // ShapefileWriter
  public static boolean addDbfHeader(Class<?> colType, String colName, DbaseFileHeader dbfHeader) throws DbaseFileException {
    // this code was 'inspired by' a similar routine in ShapefileFeatureWriter, including the supported
    // types and all the magic constants
    if ((colType == Integer.class) || (colType == Short.class) || (colType == Byte.class)) {
        dbfHeader.addColumn(colName, 'N', 9, 0);
    } else if (colType == Long.class) {
        dbfHeader.addColumn(colName, 'N', 19, 0);
    } else if (colType == BigInteger.class) {
        dbfHeader.addColumn(colName, 'N', 33, 0);
    } else if (colType == Float.class) {
        int l = 24;
        // GDAL format default is 15 decimal places of precision
        // http://www.gdal.org/drv_shapefile.html
        int d = Math.min(Math.max(24 - 2, 0), 15);
        dbfHeader.addColumn(colName, 'N', l, d);
    } else if (colType == Double.class) {
        int l = 33;
        int d = Math.min(Math.max(l - 2, 0), 15);
        dbfHeader.addColumn(colName, 'N', l, d);
    } else if (Number.class.isAssignableFrom(colType)) {
        int l = 33;
        int d = Math.max(l - 2, 0);
        dbfHeader.addColumn(colName, 'N', l, d);
        // This check has to come before the Date one or it is never reached
        // also, this field is only activated with the following system property:
        // org.geotools.shapefile.datetime=true
    } else if (java.util.Date.class.isAssignableFrom(colType)
            && Boolean.getBoolean("org.geotools.shapefile.datetime")) {
        dbfHeader.addColumn(colName, '@', 255, 0);
    } else if (java.util.Date.class.isAssignableFrom(colType)
            || Calendar.class.isAssignableFrom(colType)) {
        dbfHeader.addColumn(colName, 'D', 255, 0);
    } else if (colType == Boolean.class) {
        dbfHeader.addColumn(colName, 'L', 1, 0);
    } else if (CharSequence.class.isAssignableFrom(colType)
            || colType == java.util.UUID.class) {
        // Possible fix for GEOT-42 : ArcExplorer doesn't like 0 length
        // ensure that maxLength is at least 1
        dbfHeader.addColumn(colName, 'C', 254, 0);
    } else {
      dbfHeader.addColumn(colName, 'C', 254, 0);
      return true;
    }

    return false;
  }

}
